/*
    Copyright 2006,2007,2008,2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "ventrilo_algo.h"
#include "ventrilo_pwd_hash.h"
#include "show_dump.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close       closesocket
    #define sleep       Sleep
    #define ONESEC      1000
    #define MYPIPENAME  "\\\\.\\pipe\\ventrcon"
    #define PIPE_T      HANDLE
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <pthread.h>
    #include <fcntl.h>
    #include <errno.h>

    #define ONESEC      1
    #define stristr     strcasestr
    #define stricmp     strcasecmp
    #define strnicmp    strncasecmp
    #define MYPIPENAME  "/tmp/ventrcon_pipe"
    #define PIPE_T      int
#endif

#ifdef WIN32
    #define quick_thread(NAME, ARG) DWORD WINAPI NAME(ARG)
    #define thread_id   HANDLE
#else
    #define quick_thread(NAME, ARG) void *NAME(ARG)
    #define thread_id   pthread_t
#endif

thread_id quick_threadx(void *func, void *data) {
    thread_id   tid;
#ifdef WIN32
    DWORD   tmp;

    tid = CreateThread(NULL, 0, func, data, 0, &tmp);
    if(!tid) return(0);
#else
    if(pthread_create(&tid, NULL, func, data)) return(0);
#endif
    return(tid);
}

void quick_threadk(thread_id tid) {
#ifdef WIN32
    TerminateThread(tid, 0);
#else
    pthread_cancel(tid);
#endif
}

void quick_threadz(thread_id tid) {
#ifdef WIN32
    DWORD   ret;

    for(;;) {
        GetExitCodeThread(tid, &ret);
        if(!ret) break;
        sleep(100);
    }
#else
    pthread_join(tid, NULL);
#endif
}

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.2.9a"
#define BUFFSZ      0xffff
#define PORT        3784
#define INPUTSZ     100
#define TIMEOUT     30
#define CYCLEMS     100
#define V3CACHEFILE "ventrcon_cache.dat"
#define CLOSE_SD    { close(sd); sd = -1; } // useless stuff

#define HANDLE_INPUT \
        if(!stricmp(input_cmd, "/chan") || !stricmp(input_cmd, "/subchan") || !stricmp(input_cmd, "/user")) { \
            show_chan_help(); \
        \
        } else if(!strnicmp(input_cmd, "/chan ", 6)) { \
            if(dochan(sd, &client, buff, input_cmd, myver, 0) < 0) goto disconnect; \
        \
        } else if(!strnicmp(input_cmd, "/subchan ", 9)) { \
            if(dosubchan(sd, &client, buff, input_cmd, myver) < 0) goto disconnect; \
        \
        } else if(!strnicmp(input_cmd, "/user ", 6)) { \
            if(douser(sd, &client, buff, input_cmd) < 0) goto disconnect;



typedef struct {    // for the moment this struct is all in memory (2 megabytes)
    u8      name[33];
} chans_t;

typedef struct {
    u32     ip;
    u16     port;
    u8      v3handshake_key[256];
    u8      v3handshake[16];
    int     handshake_num;
} v3cache_t;



#include "ventrilo3_handshake.c"
int ventrilo3_handshake_cache(u32 ip, u16 port, u8 *v3handshake, int *handshake_num, u8 *v3handshake_key, int redocache);
u8 *read_pipe(u8 *buff, int buffsz, PIPE_T mypipe);
PIPE_T create_pipe(int buffsz);
quick_thread(fgets_thread, u8 **line);
int ventrecv(int sd, ventrilo_key_ctx *client, ventrilo_key_ctx *server, u8 *buff, int *admin_result);
void get_myid(u8 *data, int datasz, u16 *my_id);
int dosubchan(int sd, ventrilo_key_ctx *client, u8 *buff, u8 *input, u8 *myver);
int douser(int sd, ventrilo_key_ctx *client, u8 *buff, u8 *input);
int dochan(int sd, ventrilo_key_ctx *client, u8 *buff, u8 *input, u8 *myver, int subchan);
int doadmin(int sd, ventrilo_key_ctx *client, u8 *buff, u8 *adminpass, int ask);
int getxx(u8 *data, u32 *ret, int bits);
int putxx(u8 *data, u32 num, int bits);
int putmm(u8 *data, u8 *val, int len);
int putss(u8 *data, u8 *val, int len);
int sendventmsg(int sd, ventrilo_key_ctx *client, u8 *buff, u8 *msg, int msglen);
int delimit(u8 *data);
int ventrilo_version(u8 *myver);
int send_ventrilo(int sd, ventrilo_key_ctx *client, u8 *buff, int len);
int recv_ventrilo(int sd, ventrilo_key_ctx *server, u8 *buff);
int rndxx_official(u8 *data, int len, u32 *seed, int fixlen);
int rndxx(u8 *data, int len, u32 *seed, int fixlen);
void show_chan_help(void);
int timeout(int sock, int sec, int msec);
u32 resolv(char *host);
void std_err(void);



PIPE_T  ventrcon_pipe   = 0;
FILE    *fdlog      = NULL;
chans_t *chans      = NULL;
int     proto       = 0,
        verbose     = 0,
        adminok     = 0,
        docache     = 0;



int main(int argc, char *argv[]) {
    ventrilo_key_ctx    client,
                        server;
    thread_id   stdin_tid   = 0;
    struct  sockaddr_in peer;
    struct  linger  ling = {1,1};
    FILE    *fd             = NULL;
    u32     seed,
            wait_some_seconds = 0;
    int     sd,
            i,
            len,
            handshake_num,
            waitsec         = 0,
            admin_result    = 0,
            adminpassask    = 1,
            reconnect       = 0,
            showloginname   = 0,    // "Show login name in remote status requests"
            pipefile        = 0,
            redocache       = 0;
    u16     port            = PORT,
            my_id;
    u8      input[6 + INPUTSZ + 1] = "",
            password[33]    = "",
            phonetic[33]    = "",
            v3handshake_key[256] = "",   // do not modify!
            adminpass[33]   = "",
            current_players = 0,
            max_players     = 0,
            *buff           = NULL,
            *batchfile      = NULL,
            *quickcmd       = NULL,
            *input_cmd      = NULL,
            *v3handshake    = NULL,
            *myver          = "3.0.0",
            *nick           = " ",
            *input_line     = NULL,
            *logfile        = NULL,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    //setbuf(stdin,  NULL);
    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fputs("\n"
        "Ventrilo RCon tool "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    if(argc < 2) {
        fprintf(stderr, "\n"
            "Usage: %s <server[:port]>\n"
            "\n"
            "-p PORT   server port (%hu)\n"
            "-w PASS   force password usage (automatically asked if needed)\n"
            "-n NICK   choose nickname (default is \"%s\")\n"
            "-a PASS   force admin password (automatically asked if needed)\n"
            "          you can provide admin passwords in any moment using /login\n"
            "-c CMD    send the rcon or custom command CMD and exit\n"
            "-f FILE   execute all the commands contained in FILE, if FILE is - will be\n"
            "          used the standard input as source of the commands\n"
            "-F        create the named pipe file %s which can be used by any\n"
            "          other external program to send commands through ventrcon\n"
            "-v        debug: hex dump of any send and received packet\n"
            "-V VER    force a specific server version which is not guessed automatically\n"
            "-L        force \"Show login name in remote status requests\" (automatic)\n"
            "-R        force reconnection on disconnection\n"
            "-t SEC    seconds to wait between the sending of each command (for -f)\n"
            "-l FILE   append all the commands sent and the messages received into FILE\n"
            "-C        enable the caching of the handshakes needed in Ventrilo 3.x to avoid\n"
            "          to contact often the centralized servers in some cases\n"
            "\n", argv[0], port, nick, MYPIPENAME);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            fprintf(stderr, "\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'w': {
                sprintf(password, "%.*s", sizeof(password), argv[++i]);
                } break;
            case 'a': {
                sprintf(adminpass, "%.*s", sizeof(adminpass), argv[++i]);
                adminpassask    = 0;
                } break;
            case 'p': port          = atoi(argv[++i]);  break;
            case 'n': nick          = argv[++i];        break;
            case 'c': quickcmd      = argv[++i];        break;
            case 'f': batchfile     = argv[++i];        break;
            case 'F': pipefile      = 1;                break;
            case 'v': verbose       = 1;                break;
            case 'V': myver         = argv[++i];        break;
            case 'L': showloginname = 1;                break;
            case 'R': reconnect     = 1;                break;
            case 't': waitsec       = atoi(argv[++i]);  break;
            case 'l': logfile       = argv[++i];        break;
            case 'C': docache       = 1;                break;
            default: {
                fprintf(stderr, "\nError: wrong command-line argument (%s)\n\n", argv[i]);
                exit(1);
                } break;
        }
    }

    p = strchr(argv[argc], ':');
    if(p) {
        *p = 0;
        port = atoi(p + 1);
    }

    peer.sin_addr.s_addr = resolv(argv[argc]);
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    fprintf(stderr, "- target:   %s : %hu\n",
        inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    seed  = ~time(NULL);
    proto = ventrilo_version(myver);
    fprintf(stderr, "- use default %s version (protocol %d)\n", myver, proto);

    if(logfile) {
        fprintf(stderr, "- append output to file %s\n", logfile);
        fdlog = fopen(logfile, "ab");
        if(!fdlog) std_err();
        setbuf(fdlog, NULL);
    }

    if(batchfile) {
        if(!strcmp(batchfile, "-")) {
            fd = stdin;
        } else {
            fprintf(stderr, "- open batch file %s\n", batchfile);
            fd = fopen(batchfile, "rb");
            if(!fd) std_err();
        }
    }
    if(pipefile) {
        ventrcon_pipe = create_pipe(INPUTSZ);
    }

    buff = malloc(BUFFSZ + 1);
    if(!buff) std_err();

    fprintf(stderr, "- allocate channels memory\n");
    chans = malloc(sizeof(chans_t) * 0x10000);
    if(!chans) std_err();

redo_join:
    wait_some_seconds = (u32)time(NULL);
    adminok         = 0;
    admin_result    = 0;
    for(i = 0; i <= 0xffff; i++) {
        chans[i].name[0] = 0;
    }

    fprintf(stderr, "- connect to server\n");
    for(;;) {
        // code from ventrilofp
            if((proto >= 3) && !v3handshake) {
                fprintf(stderr, "\n- get server's hash through centralized Ventrilo servers or local cache:\n");
                v3handshake = malloc(16);
                if(ventrilo3_handshake_cache(peer.sin_addr.s_addr, ntohs(peer.sin_port), v3handshake, &handshake_num, v3handshake_key, redocache) < 0) {
                    myver = "2.3.0";
                    fprintf(stderr, "\n- Ventrilo 3 centralized handshake failed, I switch to protocol %s\n", myver);
                    proto = ventrilo_version(myver);
                    free(v3handshake);
                    v3handshake = NULL;
                } else {
                    fprintf(stderr, "- Ventrilo 3 server handshake:\n");
                    show_dump(v3handshake, 16, stderr);
                }
                //redocache = 0;  // reset
            }

            sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
            if(sd < 0) std_err();
            if(connect(sd, (struct sockaddr *)&peer, sizeof(struct sockaddr_in))
              < 0) std_err();
            fprintf(stderr, "- connected\n");
            setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));

            p = buff;
            p += putxx(p, 0,                32);
            p += putss(p, myver,            16);
            p += rndxx_official(p, 32, &seed, 1);
            p += rndxx_official(p, 32, &seed, 1);

            if(send_ventrilo(sd, NULL, buff, p - buff) < 0) goto disconnect;
            len = recv_ventrilo(sd, NULL, buff);
            if(len < 0) goto disconnect;

            if((len > 12) && (buff[8] != 4)) {
                p = strstr(buff + 12, "version ");
                if(p) {
                    myver = strdup(p + 8);
                    fprintf(stderr, "\n- server uses version %s\n", myver);
                    p = strrchr(myver, '.');
                    if(p) strcpy(p, ".0");
                    proto = ventrilo_version(myver);
                    fprintf(stderr, "- set version to %s (protocol %d)\n", myver, proto);
                    CLOSE_SD
                    continue;
                }
                fprintf(stderr, "\nError: %s\n\n", buff[12] ? (buff + 12) : (u8 *)"you have been disconnected without reason, probably you are banned");
                CLOSE_SD
                if(reconnect) goto redo_join;   // good?
                exit(1);
            }

            if(ventrilo_read_keys(&client, &server, buff + 12, len - 12) < 0) {
                fprintf(stderr, "- no keys found, the server probably uses a different protocol\n");
                exit(1);
            }

            if(proto == 1) {
            } else {
                len = recv_ventrilo(sd, &server, buff);
                if(len < 0) goto disconnect;
                if(len > 12) {
                    current_players = buff[12];
                    max_players     = buff[10];
                    fprintf(stderr, "- players %u/%u\n", current_players, max_players);
                }
            }

            p = buff;
            if(proto <= 1) {
                p += putxx(p, 0x0d,         32);
            } else if(proto == 2) {
                p += putxx(p, 0x36,         32);
            } else if(proto >= 3) {
                p += putxx(p, 0x48,         32);
            }
            p += putxx(p, 2,                32);
            p += putxx(p, 0,                32);
            if(proto == 0) {
                p += putxx(p, 0,            32);
                p += putxx(p, 0,            32);
                p += putxx(p, 0,            32);
            }

            if(proto <= 1) {
                p += putss(p, myver,        16);            // main version
                if(nick) {
                    p += putss(p, nick,     32);            // nickname
                } else {
                    p += rndxx(p, 32, &seed, 0);            // nickname
                }
                p += putss(p, phonetic,     32);            // phonetic
                p += putss(p, password,     32);            // password
                p += putss(p, "WIN32",      64);            // OS
                p += putss(p, myver,        16);            // client version
                p += putss(p, "",           48);            // ???

            } else if(proto == 2) {
                p += putxx(p, 0,            32);            // source IP
                p += putxx(p, 0,            32);            // source port
                p += putxx(p, 0,            32);            // family?

                p += putss(p, myver,        16);            // main version
                p += putss(p, "",           48);            // ???
                p += putss(p, myver,        16);            // client version
                ventrilo_pwd_hash(password, p); p += 32;    // password
                if(nick) {
                    p += putss(p, nick,     32);            // nickname
                } else {
                    p += rndxx(p, 32, &seed, 0);            // nickname
                }
                p += putss(p, phonetic,     32);            // phonetic
                p += putss(p, "WIN32",      64);            // OS

            } else if(proto >= 3) {
                p += putxx(p, peer.sin_addr.s_addr, 32);
                p += putxx(p, seed,         16);            //p += putxx(p, 127287, 32);
                p += putxx(p, showloginname,32);
                p += putxx(p, handshake_num,16);

                p += putmm(p, v3handshake,  16);
                p += putss(p, myver,        64);
                p += putss(p, myver,        16);            // client version
                ventrilo_pwd_hash(password, p); p += 32;    // password
                if(nick) {
                    p += putss(p, nick,     32);            // nickname
                } else {
                    p += rndxx(p, 32, &seed, 0);            // nickname
                }
                p += putss(p, phonetic,     32);            // phonetic
                p += putss(p, "WIN32",      64);            // OS
            }

            if(send_ventrilo(sd, &client, buff, p - buff) < 0) goto disconnect;
            len = recv_ventrilo(sd, &server, buff);
            if(len < 0) goto disconnect;

            if((buff[0] != 0x10) && (buff[0] != 0x0a) && (buff[0] != 0x52) && (buff[0] != 0x4c)) {
            //  && (buff[0] != 0x59)
                if((len > 8) && (buff[8] == 0x41)) {
                    fprintf(stderr, "- server is protected by password, insert it: ");
                    //fflush(stdin);
                    fgets(password, sizeof(password), stdin);
                    delimit(password);
                    CLOSE_SD
                    continue;
                }
                goto disconnect;
            }
        break;
    }

    my_id = 0;
    for(;;) {
        len = recv_ventrilo(sd, &server, buff);
        if(len < 0) goto disconnect;
        if(proto >= 3) {
            if(buff[0] == 0x34) break;
        } else if(proto == 2) {
            if(buff[0] == 0x33) break;
        } else {
            if(buff[0] == 0x0e) break;
        }
        get_myid(buff, len, &my_id);
    }
    if((proto >= 3) && v3handshake) {
        ventrilo3_algo_scramble(&client, v3handshake_key);
        ventrilo3_algo_scramble(&server, v3handshake_key);
    }

    fprintf(stderr, "- now you are ready to insert commands and/or the password\n");
    if(doadmin(sd, &client, buff, adminpass, adminpassask) < 0) goto disconnect;

    do {    // wait if the password is valid or not
        len = ventrecv(sd, &client, &server, buff, &admin_result);
        if(len < 0) goto disconnect;
    } while(!admin_result); // admin_result is -1 if wrong or 1 if ok, all the rest is 0

    if((admin_result < 0) && (quickcmd || batchfile || pipefile)) {
        CLOSE_SD
        fprintf(stderr, "\nError: your admin password seems invalid\n");
        exit(1);
    }

    if(quickcmd) {
        input_cmd = quickcmd;

        HANDLE_INPUT
        } else {
            if(!strnicmp(input_cmd, "/rcon ", 6)) {
                len = sprintf(input, "%.*s", 6 + INPUTSZ, input_cmd);
            } else {
                len = sprintf(input, "/rcon %.*s", INPUTSZ, input_cmd);
            }
            printf("- command: %s\n", input);
            if(sendventmsg(sd, &client, buff, input, len) < 0) goto disconnect;
        }

        wait_some_seconds = (u32)time(NULL); // it's useful to wait for a reply
        do {
            if(ventrecv(sd, &client, &server, buff, NULL) < 0) goto disconnect;
        } while(((u32)time(NULL) - wait_some_seconds) < 2);

        fprintf(stderr, "- done\n");
        CLOSE_SD
        return(0);
    }

    if(!batchfile && !pipefile) {
        stdin_tid = quick_threadx(fgets_thread, (void *)&input_line);
    }
    adminpassask = 0;   // mah
    wait_some_seconds = (u32)time(NULL) + waitsec;

    for(;;) {
        do {
            len = ventrecv(sd, &client, &server, buff, NULL);
            if(len < 0) goto disconnect;
        } while(len);

        len = 0;
        input_cmd = input + 6; // + 6 because I add "/rcon "
        if(batchfile) {
            if(((u32)time(NULL) - wait_some_seconds) >= waitsec) {
                if(!fgets(input_cmd, INPUTSZ, fd)) break;
                len = delimit(input_cmd);
                wait_some_seconds = (u32)time(NULL);
            }
        } else if(pipefile) {
            if(((u32)time(NULL) - wait_some_seconds) >= waitsec) {
                if(!read_pipe(input_cmd, INPUTSZ, ventrcon_pipe)) break;
                len = delimit(input_cmd);
                wait_some_seconds = (u32)time(NULL);
            }
        } else {
            if(input_line) {
                strcpy(input_cmd, input_line);
                input_line = NULL;
                len = strlen(input_cmd);
            }
        }
        if(!len) continue;
        if(len < 0) break;

        if(batchfile || pipefile) {
            printf("\n> %s\n", input_cmd);
        } else {
            printf("\n");
        }
        if(fdlog) fprintf(fdlog, "\n> %s\r\n", input_cmd);

        HANDLE_INPUT
        } else if(!strnicmp(input_cmd, "/login", 6)) {
            if(doadmin(sd, &client, buff, adminpass, 1) < 0) goto disconnect;

        } else if(!strnicmp(input_cmd, "/rcon ", 6)) {
            if(sendventmsg(sd, &client, buff, input_cmd, len) < 0) goto disconnect;

        } else {
            memcpy(input, "/rcon ", 6);
            len += 6;
            if(sendventmsg(sd, &client, buff, input, len) < 0) goto disconnect;
        }

        continue;

disconnect:
        fprintf(stderr, "\n- alert: connection interrupted by the server\n");
        CLOSE_SD
        if(!adminok && !showloginname) {
            fprintf(stderr, "- I try to enable the \"Show login name in remote status requests\" option\n");
            showloginname = 1;
            goto redo_join;
        }
        if(docache && !redocache) {
            free(v3handshake);
            v3handshake = NULL;
            redocache = 1;
            showloginname = 0;
            goto redo_join;
        }
        if(reconnect) break;
        fprintf(stderr,
            "  probably it uses a different protocol or you are banned or something else.\n"
            "  if your server uses my ventrilobotomyfix use -V VERSION\n");
        exit(1);
    }

    //showloginname = 0;
    redocache     = 0;
    if(sd >= 0) CLOSE_SD
    //if(stdin_tid) {
        //quick_threadk(stdin_tid);
        //quick_threadz(stdin_tid);
    //}
    if(reconnect) {
        sleep(ONESEC);
        goto redo_join;     // if we are using a batch file it will not be reloaded from the beginning if disconnected, I think it's better
    }

    if(fd && (fd != stdin)) fclose(fd);
    if(fdlog) fclose(fdlog);
    fprintf(stderr, "- done\n");
    return(0);
}



int ventrilo3_handshake_cache(u32 ip, u16 port, u8 *v3handshake, int *handshake_num, u8 *v3handshake_key, int redocache) {
    v3cache_t   v3cache;
    static FILE *fd = NULL;
    int     ret;

    memset(v3handshake, 0, sizeof(v3cache.v3handshake));    // not useful but ok
    memset(v3handshake_key, 0, sizeof(v3cache.v3handshake_key));

    if(docache && !redocache) { // read from the cache
        if(!fd) {
            fd = fopen(V3CACHEFILE, "r+b");
            if(!fd) fd = fopen(V3CACHEFILE, "w+b");
            if(!fd) std_err();
        }

        fseek(fd, 0, SEEK_SET);
        while(fread(&v3cache, sizeof(v3cache_t), 1, fd)) {
            if((v3cache.ip == ip) && (v3cache.port == port)) {
                memcpy(v3handshake, v3cache.v3handshake, sizeof(v3cache.v3handshake));
                *handshake_num = v3cache.handshake_num;
                memcpy(v3handshake_key, v3cache.v3handshake_key, sizeof(v3cache.v3handshake_key));
                return(0);
            }
        }
    }
    if(docache) {
        fprintf(stderr, "- update the cache file\n");
    }

    ret = ventrilo3_handshake(ip, port, v3handshake, handshake_num, v3handshake_key);
    if(!ret && docache) {
        fseek(fd, 0, SEEK_SET);
        while(fread(&v3cache, sizeof(v3cache_t), 1, fd)) {
            if((v3cache.ip == ip) && (v3cache.port == port)) {
                fseek(fd, -sizeof(v3cache_t), SEEK_CUR);    // update the current one
                break;
            }
        }
        v3cache.ip   = ip;
        v3cache.port = port;
        memcpy(v3cache.v3handshake, v3handshake, sizeof(v3cache.v3handshake));
        v3cache.handshake_num = *handshake_num;
        memcpy(v3cache.v3handshake_key, v3handshake_key, sizeof(v3cache.v3handshake_key));
        fwrite(&v3cache, sizeof(v3cache_t), 1, fd);
        fflush(fd);
    }
    return(ret);
}



u8 *read_pipe(u8 *buff, int buffsz, PIPE_T mypipe) {
    int     len = 0;

#ifdef WIN32
    DWORD   tmp;
    while(ConnectNamedPipe(mypipe, NULL)) {
        while(ReadFile(mypipe, buff + len, 1, &tmp, NULL)) {
            len++; // as below
            if((buff[len - 1] == '\n') || (len >= buffsz)) {
                buff[len] = 0;
                DisconnectNamedPipe(mypipe);    // needed!
                return(buff);
            }
        }
        DisconnectNamedPipe(mypipe);
    }
#else
    for(;;) {
        if(read(mypipe, buff + len, 1) != 1) break;
            len++; // as above
            if((buff[len - 1] == '\n') || (len >= buffsz)) {
                buff[len] = 0;
                return(buff);
            }
    }
#endif
    return(NULL);
}



PIPE_T create_pipe(int buffsz) {
    PIPE_T  mypipe;

    printf("- create named pipe file: %s\n", MYPIPENAME);
#ifdef WIN32
    mypipe = CreateNamedPipe(
        MYPIPENAME,
        PIPE_ACCESS_DUPLEX,
        PIPE_TYPE_BYTE | PIPE_READMODE_BYTE | PIPE_WAIT,
        PIPE_UNLIMITED_INSTANCES,
        buffsz,
        buffsz,
        NMPWAIT_USE_DEFAULT_WAIT,
        NULL);
    if(mypipe == INVALID_HANDLE_VALUE) {
        fprintf(stderr, "\nError: unable to create the named pipe %s\n", MYPIPENAME);
        exit(1);
    }
#else
    mypipe = mkfifo(MYPIPENAME, 0755);
    if((mypipe < 0) && (errno != EEXIST)) {
        fprintf(stderr, "\nError: unable to create the named pipe %s\n", MYPIPENAME);
        exit(1);
    }
    mypipe = open(MYPIPENAME, O_RDWR);
    if(mypipe < 0) {
        fprintf(stderr, "\nError: unable to open the named pipe %s\n", MYPIPENAME);
        exit(1);
    }
#endif
    return(mypipe);
}



quick_thread(fgets_thread, u8 **line) { // this solution is better then using kbhit/getch where I should handle anything
    static u8   input[INPUTSZ + 1];

    *line = NULL;
    for(;;) {
        if(!fgets(input, INPUTSZ, stdin)) break;
        if(delimit(input) > 0) *line = input;
    }
    return(0);
}



int ventrecv(int sd, ventrilo_key_ctx *client, ventrilo_key_ctx *server, u8 *buff, int *admin_result) {
    int     i,
            len,
            off;

    if(admin_result) *admin_result = 0;
    if(timeout(sd, 0, CYCLEMS) < 0) return(0);
    len = recv_ventrilo(sd, server, buff);
    if(len < 0) return(-1);
    if(!len) return(0);
    off = len;  // in case of missed values later
    if((buff[0] == 0x5a) || (buff[0] == 0x63)) {    // ping/pong, useless
        if(send_ventrilo(sd, client, buff, len) < 0) return(-1);
        return(0);
    }
    if(((proto >= 3) && (buff[0] == 0x59)) || (buff[0] == 0x3a)) {
        if(adminok) {
            fprintf(stderr, "- command not accepted or other errors\n");
        } else {
            fprintf(stderr, "- WRONG admin password, use /login to re-enter the password\n");
        }
        if(admin_result) *admin_result = -1;
        return(0);
    }
    if(buff[0] == 0x06) {
        if(buff[8] == 0x02) {
            if(admin_result) *admin_result = 1;
            adminok = 1;
            show_chan_help();
            return(0);
        } else if(buff[8] == 0x04) {
            // server keys
        } else {
            off = 12;
        }
    } else {
        if(proto >= 3) {
            if(buff[0] != 0x42) return(0);
            off = 14;
        } else if(proto == 2) {
            if(buff[0] != 0x43) return(0);
            off = 14;
        } else {
            if(buff[0] != 0x07) return(0);
            off = 12;
        }
    }
    len -= off;
    if(len <= 0) return(0);
    buff += off;
    for(i = 0; i < len; i++) {  // check if there is readable data to visualize
        if(!(!buff[i] || (buff[i] == ' ') || (buff[i] == '\t'))) break;
    }
    if(i < len) {
        printf("%.*s\n", len, buff);
        if(fdlog) fprintf(fdlog, "%.*s\r\n", len, buff);
    }
    return(len);
}



u8 *next(u8 **data) {
    u8      *ret,
            *p;

    ret = *data;
    p   = *data;
    while(*p && ((*p == ' ') || (*p == '\t'))) p++;
    if(*p == '\"') {
        ret++;
        p++;
        while(*p) {
            if(*p == '\"') {
                *p++ = 0;
                break;
            }
            p++;
        }
    } else {
        while(*p && (*p != ' ') && (*p != '\t')) p++;
    }
    if(*p) {
        *p++ = 0;
        while(*p && ((*p == ' ') || (*p == '\t'))) p++;
    }
    *data = p;
    return(ret);
}



int dosubchan(int sd, ventrilo_key_ctx *client, u8 *buff, u8 *input, u8 *myver) {
    int     i,
            channum;
    u8      *chancmd,
            *subchan,
            *p;

    p = input;
    chancmd     = next(&p);
    subchan     = next(&p);

    for(i = 0; i <= 0xffff; i++) {
        if(!stricmp(subchan, chans[i].name)) break;
    }
    if(i <= 0xffff) {
        channum = i;
    } else {
        channum = atoi(subchan);
    }

    memset(input, ' ', p - input);
    memcpy(input, "/ create", 8);

    return(dochan(sd, client, buff, input, myver, channum));
}



int douser(int sd, ventrilo_key_ctx *client, u8 *buff, u8 *input) {
    int     i,
            len;
    u8      *usercmd,
            *cmd,
            *username,
            *password,
            *p;

    p = input;
    usercmd     = next(&p);
    cmd         = next(&p);
    username    = next(&p);
    password    = next(&p);
    //admin       = next(&p); // not implemented for security reasons

    if(!username[0]) {
        fprintf(stderr, "- a required parameter for the /chan command is missed or wrong\n");
        return(0);
    }

    if(!stricmp(cmd, "create")) {
        printf("- create user %s\n", username);
    } else {
        fprintf(stderr, "- wrong /user command\n");
        return(0);
    }

    p = buff;
    if(proto >= 3) {
        p += putxx(p, 0x4a,         32);
        p += putxx(p, 0,            16);    // open the user editor
        p += putxx(p, 0,            32);
        p += putxx(p, 0,            16);
        p += putxx(p, 0,            32);
        p += putxx(p, 0,            32);
        if(send_ventrilo(sd, client, buff, p - buff) < 0) return(-1);

        p = buff;
        p += putxx(p, 0x4a,         32);
        p += putxx(p, 1,            16);    // 1 = create, 2 = delete
        p += putxx(p, 0,            32);
        p += putxx(p, 1,            16);
        p += putxx(p, 0,            32);
        p += putxx(p, 0,            32);
        p += putxx(p, 0,            32);    // id of the user in delete mode
        ventrilo_pwd_hash(password, p); p += 32;
        p += putxx(p, 0,            32);
        for(i = 0; i < 64; i++) *p++ = 0;   // no rights (admin requires 1 at offset 16 from here)
        len = strlen(username);     p += putbe(p, len, 16); p += putss(p, username, len);
        p += putbe(p, 0, 16);               // string?
        p += putbe(p, 0, 16);               // notes
        p += putbe(p, 0, 16);               // locked
        p += putbe(p, 0, 16);               // string?
        p += putbe(p, 0, 16);               // string?
        if(send_ventrilo(sd, client, buff, p - buff) < 0) return(-1);

        p = buff;
        p += putxx(p, 0x4a,         32);
        p += putxx(p, 4,            16);    // close the user editor
        p += putxx(p, 0,            32);
        p += putxx(p, 0,            16);
        p += putxx(p, 0,            32);
        p += putxx(p, 0,            32);
    } else {
        if(proto == 2) {
            p += putxx(p, 0x5e,     32);
        } else {
            p += putxx(p, 0x1b,     32);
        }
        p += putxx(p, 1,            16);    // 1 = create, 2 = delete
        p += putxx(p, 1,            32);
        p += putxx(p, 0,            16);
        p += putxx(p, 0,            32);
        p += putxx(p, 0,            32);
        p += putss(p, username,     32);
        if(proto <= 1) {
            p += putss(p, password, 32);
        } else {
            ventrilo_pwd_hash(password, p); p += 32;
        }
    }
    return(send_ventrilo(sd, client, buff, p - buff));
}



int dochan(int sd, ventrilo_key_ctx *client, u8 *buff, u8 *input, u8 *myver, int subchan) {
    int     maj,
            min,
            i,
            len,
            channum;
    u8      *chancmd,
            *cmd,
            *displayname,
            *phonetic,
            *chanpass,
            *adminpass,
            *comment,
            *p;

    p = input;
    chancmd     = next(&p);
    cmd         = next(&p);
    displayname = next(&p);
    phonetic    = next(&p);
    chanpass    = next(&p);
    adminpass   = next(&p);
    comment     = next(&p);

    sscanf(myver, "%d.%d", &maj, &min);

    if(!stricmp(cmd, "list")) {
        printf(
            "  ID    channel_name\n"
            "  --------------------------------------\n");
        for(i = 0; i <= 0xffff; i++) {
            if(chans[i].name[0]) printf("  %-5d %.32s\n", i, chans[i].name);
        }
        return(0);
    }

    if(!displayname[0]) {
        fprintf(stderr, "- a required parameter for the /chan command is missed or wrong\n");
        return(0);
    }

    if(!stricmp(cmd, "create")) {
        channum = 0;
        printf("- create chan %s\n", displayname);
    } else if(!stricmp(cmd, "delete")) {
        for(i = 0; i <= 0xffff; i++) {
            if(!stricmp(displayname, chans[i].name)) break;
        }
        if(i <= 0xffff) {
            channum = i;
        } else {
            channum = atoi(displayname);
        }
        displayname = "";
        printf("- delete chan %d (%s)\n", channum, chans[channum].name);
        if((channum >= 0) && (channum <= 0xffff)) chans[channum].name[0] = 0;
    } else {
        fprintf(stderr, "- wrong /chan command\n");
        return(0);
    }

    p = buff;
    if(proto >= 3) {
        p += putxx(p, 0x49,         32);
        p += putxx(p, 0,            16);
        if(!stricmp(cmd, "delete")) {
            p += putxx(p, 2,        16);
            memset(p, 0, 80);
            putxx(p + 32, channum,  16);
            p += 80;
        } else {
            if(!stricmp(cmd, "create")) {
                p += putxx(p, 1,    16);
            } else if(!stricmp(cmd, "edit")) {
                p += putxx(p, 5,    16);
            }
            ventrilo_pwd_hash(chanpass, p); p += 32;
            p += putxx(p, channum,  16);
            p += putxx(p, subchan,  16);
            p += putxx(p, 0,        32);
            p += putxx(p, 1,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 0,        16);
            p += putxx(p, 0,        16);
            p += putxx(p, 0,        16);
            p += putxx(p, 0,        16);
            p += putxx(p, 1,        16);    // ???
            p += putxx(p, 0,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 0,        16);
            if(chanpass[0]) {
                p += putxx(p, 1,    16);    // 2 for user authorization
            } else {
                p += putxx(p, 0,    16);
            }
            p += putxx(p, 0,        16);
            p += putxx(p, -1,       32);
            p += putxx(p, 0,        32);
            len = strlen(displayname);  p += putbe(p, len, 16); p += putss(p, displayname, len);
            len = strlen(phonetic);     p += putbe(p, len, 16); p += putss(p, phonetic,    len);
            len = strlen(comment);      p += putbe(p, len, 16); p += putss(p, comment,     len);
        }
    } else {
        if(proto == 2) {
            p += putxx(p, 0x46,     32);
        } else {
            p += putxx(p, 0x0a,     32);
        }
        p += putxx(p, 0,            16);
        p += putxx(p, channum,      16);
        if(!stricmp(cmd, "create")) {
            p += putxx(p, 1,        16);
        } else if(!stricmp(cmd, "delete")) {
            p += putxx(p, 2,        16);
        } else if(!stricmp(cmd, "edit")) {
            p += putxx(p, 5,        16);
        }
        p += putxx(p, subchan,      16);
        p += putxx(p, 0,            32);
        p += putss(p, displayname,  32);
        p += putss(p, phonetic,     32);
        if(proto <= 1) {
            p += putss(p, chanpass, 32);
            p += putss(p, adminpass,32);
        } else {
            ventrilo_pwd_hash(chanpass, p); p += 32;
            ventrilo_pwd_hash(adminpass,p); p += 32;
        }
        if((maj >= 2) && (min >= 2)) {
            p += putss(p, comment,  64);
            p += putxx(p, 1,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 1,        16);
            p += putxx(p, 0,        16);
            p += putxx(p, 0,        16);
            p += putxx(p, 0,        16);
            p += putxx(p, 0,        16);
            if(!((maj == 2) && (min == 2))) {
                p += putxx(p, 0,    16);
                p += putxx(p, 0,    16);
            }
        }
    }
    return(send_ventrilo(sd, client, buff, p - buff));
}



void get_myid(u8 *data, int datasz, u16 *my_id) {
    u32     num;
    u8      *p,
            *l;

    num = 0;
    p = data;
    l = data + datasz;
    if(proto >= 3) {
        if((data[0] != 0x5d) || (data[4] != 0x04) || (data[6] != 0x02)) return;
        p += 16;
        p += getbe(p, &num, 16);    // Server 1
        p += num;
        if(p >= l) return;
        p += getbe(p, &num, 16);    // Server 1
        p += num;
        p += 6;
        if(p >= l) return;
        getxx(p, &num, 16);
    } else if(proto == 2) {
        if((data[0] != 0x44) || (data[4] != 0x04) || (data[6] != 0x01)) return;
        getxx(p + 8, &num, 16);
    } else {
        if((data[0] != 0x04) || (data[4] != 0x04) || (data[6] != 0x01)) return;
        getxx(p + 8, &num, 16);
    }
    if(num) *my_id = num;
}



int doadmin(int sd, ventrilo_key_ctx *client, u8 *buff, u8 *adminpass, int ask) {
    u8      *p;

    adminok = 0;

    if(ask) {
        fprintf(stderr, "- insert the admin password: ");
        //fflush(stdin);
        fgets(adminpass, 33, stdin);
        delimit(adminpass);
    }

    p = buff;
    if(proto >= 3) {
        p += putxx(p, 0x63, 32);
    } else if(proto == 2) {
        p += putxx(p, 0x54, 32);
    } else {
        p += putxx(p, 0x0b, 32);
    }
    p += putxx(p, 0,        16);
    p += putxx(p, 0,        16);
    p += putxx(p, 0,        32);
    if(proto >= 2) {
        ventrilo_pwd_hash(adminpass, p); p += 32;
    } else {
        p += putmm(p, adminpass, 32);
    }
    p += putss(p, "",       32);
    if(proto >= 3) {
        memset(p, 0, 64);
        p += 64;
    }
    return(send_ventrilo(sd, client, buff, p - buff));
}



int getxx(u8 *data, u32 *ret, int bits) {
    u32     num;
    int     i,
            bytes;

    bytes = bits >> 3;
    for(num = i = 0; i < bytes; i++) {
        num |= (data[i] << (i << 3));
    }
    *ret = num;
    return(bytes);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;
    for(i = 0; i < bytes; i++) {
        data[i] = num >> (i << 3);
    }
    return(bytes);
}



int putmm(u8 *data, u8 *val, int len) {
    memcpy(data, val, len);
    return(len);
}



int putss(u8 *data, u8 *val, int len) {
    strncpy(data, val, len);
    return(len);
}



int sendventmsg(int sd, ventrilo_key_ctx *client, u8 *buff, u8 *msg, int msglen) {
    u8      *p;

    p = buff;
    if(proto >= 3) {
        p += putxx(p, 0x42, 32);
    } else if(proto == 2) {
        p += putxx(p, 0x43, 32);
    } else {
        p += putxx(p, 0x07, 32);
    }
    p += putxx(p, 0,        16);
    p += putxx(p, 2,        16);
    p += putxx(p, 0,        32);
    if(proto >= 2) {
        p += putbe(p, msglen,   16);
        p += putss(p, msg,      msglen);
    } else {
        p += putss(p, msg,      msglen);
        p += putxx(p, 0,        8);
    }
    return(send_ventrilo(sd, client, buff, p - buff));
}



int delimit(u8 *data) {
    u8      *p;

    for(p = data; *p && (*p != '\r') && (*p != '\n'); p++);
    *p = 0;
    return(p - data);
}



int ventrilo_version(u8 *myver) {
    int     maj,
            min,
            protocol = 0;

    sscanf(myver, "%d.%d", &maj, &min);
    if(((maj == 2) && (min < 2)) || (maj < 2)) {
        protocol = 1;
        fprintf(stderr, "- activated old protocol\n");

    } else if((maj >= 2) && (min >= 3)) {
        protocol = 2;
        fprintf(stderr, "- activated 2.3 protocol\n");

    } else if((maj >= 3)) {
        protocol = 3;
        fprintf(stderr, "- activated 3.x protocol\n");

    } else {
        fprintf(stderr, "- activated 2.2 protocol\n");
    }
    return(protocol);
}



int send_ventrilo(int sd, ventrilo_key_ctx *client, u8 *buff, int len) {
    static u8   *fastsend   = NULL;

    if(!fastsend) {
        fastsend = malloc(2 + BUFFSZ);
        if(!fastsend) std_err();
    }

    if(verbose) {
        printf("\n- send:\n");
        show_dump(buff, len, stdout);
    }

    if(client) {
        ventrilo_enc(client, buff, len);
    } else {
        ventrilo_first_enc(buff, len);
    }

    fastsend[0] = len >> 8;
    fastsend[1] = len;
    memcpy(fastsend + 2, buff, len);
    len += 2;
    if(send(sd, fastsend, len, 0) != len) return(-1);
    return(0);
}



int recv_ventrilo(int sd, ventrilo_key_ctx *server, u8 *buff) {
    int     t;
    u32     tmp;
    u16     rem,
            len;
    u8      *p,
            *l;

    if(timeout(sd, TIMEOUT, 0) < 0) return(-1);
    if(recv(sd, (void *)&len, 1, 0)     < 0)  return(-1);
    if(timeout(sd, TIMEOUT, 0) < 0) return(-1);
    if(recv(sd, (void *)&len + 1, 1, 0) < 0)  return(-1);

    len = ntohs(len);
    if(!len) return(-1);    // uhmmm ok?
    rem = len;
    p   = buff;

    while(rem) {
        if(timeout(sd, TIMEOUT, 0) < 0) return(-1);
        t = recv(sd, p, rem, 0);
        if(t <= 0) return(-1);
        p   += t;
        rem -= t;
    }
    buff[len] = 0;  // rarely useful

    if(server) {
        ventrilo_dec(server, buff, len);
    } else {
        ventrilo_first_dec(buff, len);
    }

    if(verbose) {
        printf("\n- recv:\n");
        show_dump(buff, len, stdout);
    }

    if((proto >= 3) && (buff[0] == 0x60)) {     // channels indexing here
        l = buff + len;
        for(p = buff + 8; p < l;) {
            p += getxx(p, &tmp, 16);
            p += 46;
            p += getxx(p, &t, 16);
            t = ntohs(t);
            if(t > 32) break;
            strncpy(chans[tmp].name, p, t);
            chans[tmp].name[t] = 0;
            p += t;
            if(p > l) break;
            p += getxx(p, &t, 16);              // phonetic
            p += ntohs(t);
            if(p > l) break;
            p += getxx(p, &t, 16);              // comments
            p += ntohs(t);
        }
    }
    if((proto >= 3) && (buff[0] == 0x49)) {     // channels just created
        if(len >= 90) {
            getxx(buff + 40, &tmp, 16);
            p = buff + 88;
            p += getxx(p, &t, 16);
            t = ntohs(t);
            if(t <= 32) {
                if(buff[6] == 1) {              // add chan
                    strncpy(chans[tmp].name, p, t);
                    chans[tmp].name[t] = 0;
                } else if(buff[6] == 2) {       // remove chan
                    chans[tmp].name[0] = 0;
                }
            }
        }
    }
    if((buff[0] == 10) || (buff[0] == 0x46)) {
        if(len >= 48) {
            getxx(buff + 6, &tmp, 16);
            if(buff[8] == 1) {                  // add chan
                strncpy(chans[tmp].name, buff + 16, 32);
            } else if(buff[8] == 2) {           // remove chan
                chans[tmp].name[0] = 0;
            }
        }
    }

    return(len);
}



    // the following function is totally useless, I leave it only for having a 100% compatibility with Ventrilo in how it generates random chars... I repeat, useless
int rndxx_official(u8 *data, int len, u32 *seed, int fixlen) {
    static const char avoid[] = ",#'`-+._|/=:\"\\"; // from Ventrilo
    u32     rnd,
            size;
    u8      *p;

    rnd = *seed;
    size = len;

    if(!fixlen) {
        len = rnd % len;
        if(len < 3) len = 3;
    }

    for(p = data;;) {
        rnd = ((rnd * 0x343FD) + 0x269EC3) & 0x7f;
        // rnd = (((rnd * 0x343FD) + 0x269EC3) >> 1) & 0x7f;    // now less official but more random
        if((rnd < 0x21) || (rnd > 0x7f)) continue;
        if(strchr(avoid, rnd)) continue;
        if(!--len) break;
        *p++ = rnd;
    }

    for(len = p - data; len < size; len++) {
        *p++ = 0;
    }

    *seed = rnd;
    return(p - data);
}



int rndxx(u8 *data, int len, u32 *seed, int fixlen) {
    static const char table[] =
            "0123456789"
            "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
            "abcdefghijklmnopqrstuvwxyz";
    u32     rnd,
            size;
    u8      *p;

    rnd = *seed;
    size = len;

    if(!fixlen) {
        len = rnd % len;
        if(len < 3) len = 3;
    }

    for(p = data;;) {
        rnd = ((rnd * 0x343FD) + 0x269EC3) >> 1;
        if(!--len) break;
        *p++ = table[rnd % (sizeof(table) - 1)];
    }

    for(len = p - data; len < size; len++) {
        *p++ = 0;
    }

    *seed = rnd;
    return(p - data);
}



void show_chan_help(void) {
    fprintf(stderr,
        "- admin password OK!\n"
        "  ventrcon's custom commands:\n"
        "    /login\n"
        "        redo the admin login in any moment\n"
        "    /chan create displayname [phonetic] [password] [adminpass] [comment]\n"
        "        e.g.: /chan create mychan \"\" \"\" \"\" \"\"\n"
        "              /chan create \"this is my channel\"\n"
        "    /chan list\n"
        "        returns the full list of current channels and their IDs\n"
        "    /chan delete channel OR channel_ID\n"
        "        you can specify the name of the channel you want to delete or its ID\n"
        "    /subchan mainchan displayname [phonetic] [password] [adminpass] [comment]\n"
        "        as above but creates a subchan in the mainchan channel (name or ID)\n"
        "        subchans are just normal chans so delete or list them as below\n"
        "    /user create username password\n"
        "        create a new user, this command works from Ventrilo 2.2\n"
        "\n"
        "  now type \"help\" for the list of commands and start to manage your server:\n");
}



int timeout(int sock, int sec, int msec) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = sec;
    tout.tv_usec = msec * 1000;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            fprintf(stderr, "\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


