/*
    Copyright 2004,2005,2006,2007,2008,2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "ventrilo_algo.h"
#include "show_dump.h"
#include "acpdump2.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close       closesocket
    #define sleep       Sleep
    #define in_addr_t   uint32_t
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netdb.h>
    #include <net/if.h>
    #include <pthread.h>
#endif

#ifdef WIN32
    #define quick_thread(NAME, ARG) DWORD WINAPI NAME(ARG)
    #define thread_id   DWORD
#else
    #define quick_thread(NAME, ARG) void *NAME(ARG)
    #define thread_id   pthread_t
#endif

thread_id quick_threadx(void *func, void *data) {
    thread_id       tid;
#ifdef WIN32
    if(!CreateThread(NULL, 0, func, data, 0, &tid)) return(0);
#else
    pthread_attr_t  attr;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    if(pthread_create(&tid, &attr, func, data)) return(0);
#endif
    return(tid);
}

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.3.3"
#define BUFFSZ      0xffff
#define PORT        3784



#pragma pack(2)
typedef struct {
    uint32_t    sip;
    uint16_t    sport;
    uint32_t    dip;
    uint16_t    dport;
    uint32_t    seq1;
    uint32_t    ack1;
    uint32_t    seq2;
    uint32_t    ack2;
} tcp_info_t;
#pragma pack()



#define V3HPROXY
#include "ventrilo3_handshake.c"
void check_v3(void);
quick_thread(udpproxy, void);
quick_thread(ventrilo_proxy, int sock);
int proxy_ventrilo(int in, int out, u8 *buff);
void show_it(u8 *str, u8 *buff, int len, tcp_info_t *tcp_info);
u32 get_peer_ip_port(int sd, u16 *port);
u32 resolv(char *host);
void std_err(void);



struct  sockaddr_in peer_conn,
                    peer_bind;
FILE    *fdlog      = NULL,
        *fdloga     = NULL;
int     v3server    = 0;



int main(int argc, char *argv[]) {
    int     sdl,
            sda,
            i,
            on      = 1,
            psz;
    u16     port    = PORT,
            lport   = PORT;
    u8      *host,
            *flog   = NULL,
            *floga  = NULL;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Ventrilo proxy data decrypter "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <server>\n"
            "\n"
            "Options:\n"
            "-p PORT   server's port to which connect (%d)\n"
            "-l PORT   local port bound by this proxy (%d)\n"
            "-d FILE   all the decoded data will be stored (appended) in the file FILE\n"
            "          instead of showed on the screen, ever in hex format\n"
            "-a FILE   as above but the file will have the tcpdump/pcap format\n"
            "          note that the 16 bit number before each packet which specifies its\n"
            "          length will NOT be dumped to avoid confusion in who reads this dump\n"
            "\n"
            "How to use:\n"
            "1) launch this tool specifying the server in which you want to join\n"
            "2) open your Ventrilo client\n"
            "3) connect your client to this host (127.0.0.1 for example) on port %d\n"
            "\n", argv[0], port, lport, lport);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 'p': port  = atoi(argv[++i]);  break;
            case 'l': lport = atoi(argv[++i]);  break;
            case 'd': flog  = argv[++i];        break;
            case 'a': floga = argv[++i];        break;
            default: {
                printf("\nError: wrong command-line argument (%s)\n", argv[i]);
                exit(1);
                }
        }
    }
    host = argv[argc];

    printf("- resolv hostname %s\n", host);
    peer_conn.sin_addr.s_addr  = resolv(host);
    peer_conn.sin_port         = htons(port);
    peer_conn.sin_family       = AF_INET;
    printf("- server    %s:%hu\n", inet_ntoa(peer_conn.sin_addr), ntohs(peer_conn.sin_port));

    peer_bind.sin_addr.s_addr = INADDR_ANY;
    peer_bind.sin_port        = htons(lport);
    peer_bind.sin_family      = AF_INET;
    printf("- bind port %hu\n", ntohs(peer_bind.sin_port));

    if((peer_conn.sin_addr.s_addr == peer_bind.sin_addr.s_addr) && (peer_conn.sin_port == peer_bind.sin_port)) {
        printf("\n"
            "Error: you can't bind the same local port of the local server\n"
            "       use -l PORT or specify an external server or another interface\n");
        exit(1);
    }

    check_v3();

    if(v3server && ((peer_conn.sin_addr.s_addr == peer_bind.sin_addr.s_addr) || (port != lport))) {
        printf("\n"
            "Error: there are problems running this proxy with a local Ventrilo 3 server\n"
            "       or with the bound port different than the server's one, if server\n"
            "       uses (for example) port 1234 you must use the same with this proxy too\n");
        exit(1);
    }

    sdl = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sdl < 0) std_err();
    if(setsockopt(sdl, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
      < 0) std_err();
    if(bind(sdl, (struct sockaddr *)&peer_bind, sizeof(struct sockaddr_in))
      < 0) std_err();
    if(listen(sdl, SOMAXCONN)
      < 0) std_err();

    if(floga) {
        printf("- create/append tcpdump file %s\n", floga);
        fdloga = fopen(floga, "r+b");
        if(!fdloga) {
            fdloga = fopen(floga, "wb");
            if(!fdloga) std_err();
            create_acp(fdloga);
        } else {
            fflush(fdloga);
            fseek(fdloga, 0, SEEK_END);
        }
    }

    printf("- activate UDP proxy\n");
    quick_threadx(udpproxy, NULL);

    fdlog = stdout;
    if(flog) {
        printf("- create/append log file %s\n", flog);
        fdlog = fopen(flog, "ab");
        if(!fdlog) std_err();
    }

    printf("- ready:\n");
    for(;;) {
        psz = sizeof(struct sockaddr_in);
        sda = accept(sdl, (struct sockaddr *)&peer_bind, &psz);
        if(sda < 0) std_err();

        fprintf(fdlog,
            "- new client %s:%hu\n",
            inet_ntoa(peer_bind.sin_addr),
            ntohs(peer_bind.sin_port));

        if(!quick_threadx(ventrilo_proxy, (void *)sda)) close(sda);
    }

    close(sdl);
    return(0);
}



void check_v3(void) {
    int     i,
            sd,
            len,
            tmp;
    u8      buff[1024];

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    printf("- check if server is version 3\n");
    memset(buff, 0, 200);
    buff[4]  = 'U';
    buff[5]  = 'D';
    buff[6]  = 'C';
    buff[7]  = 'L';
    buff[9]  = 4;
    buff[11] = 200;
    buff[12] = 2;
    if(sendto(sd, buff, 200, 0, (struct sockaddr *)&peer_conn,  sizeof(struct sockaddr_in)) < 0) goto quit;
    if(v3timeout(sd, 1) < 0) {
        if(sendto(sd, buff, 200, 0, (struct sockaddr *)&peer_conn,  sizeof(struct sockaddr_in)) < 0) goto quit;
        if(v3timeout(sd, 2) < 0) goto quit;
    }
    len = recvfrom(sd, buff, sizeof(buff), 0, NULL, NULL);
    if(len < 0) goto quit;

    if(buff[168] < '3') goto quit;

    v3server = 1;
    printf("- retrieve server's online handshakes (Ventrilo 3)\n");
    ventrilo3_handshake(peer_conn.sin_addr.s_addr, ntohs(peer_conn.sin_port), NULL, &tmp, NULL);
    printf("- Ventrilo 3 server handshakes:\n");
    for(i = 0; ventrilo3_auth[i].host; i++) {
        show_dump(ventrilo3_auth[i].handshake, 16, stdout);
    }

quit:
    close(sd);
}



quick_thread(udpproxy, void) {
    struct  sockaddr_in peeru;
    int     sd,
            len,
            psz;
    u8      *buff;

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    if(bind(sd, (struct sockaddr *)&peer_bind, sizeof(struct sockaddr_in))
      < 0) std_err();

    buff = malloc(BUFFSZ);
    if(!buff) std_err();
    memcpy(&peeru, &peer_bind, sizeof(struct sockaddr_in));

    for(;;) {
        psz = sizeof(struct sockaddr_in);
        len = recvfrom(sd, buff, BUFFSZ, 0, (struct sockaddr *)&peeru, &psz);
        if(len < 0) continue;

        if((peeru.sin_addr.s_addr == peer_conn.sin_addr.s_addr) && (peeru.sin_port == peer_conn.sin_port)) {
            sendto(sd, buff, len, 0, (struct sockaddr *)&peer_bind, sizeof(struct sockaddr_in));

            if(fdloga) {
                acp_dump(
                    fdloga, SOCK_DGRAM, IPPROTO_UDP,
                    peer_conn.sin_addr.s_addr, peer_conn.sin_port, peer_bind.sin_addr.s_addr, peer_bind.sin_port,
                    buff, len,
                    NULL, NULL, NULL, NULL);
                fflush(fdloga);
            }

        } else {
            memcpy(&peer_bind, &peeru, sizeof(struct sockaddr_in));
            sendto(sd, buff, len, 0, (struct sockaddr *)&peer_conn,  sizeof(struct sockaddr_in));

            if(fdloga) {
                acp_dump(
                    fdloga, SOCK_DGRAM, IPPROTO_UDP,
                    peer_bind.sin_addr.s_addr, peer_bind.sin_port, peer_conn.sin_addr.s_addr, peer_conn.sin_port,
                    buff, len,
                    NULL, NULL, NULL, NULL);
                fflush(fdloga);
            }
        }
    }

    close(sd);
    free(buff);
    return(0);
}



quick_thread(ventrilo_proxy, int sock) {
    ventrilo_key_ctx    client,
                        server;
    tcp_info_t  tcp_info;
    fd_set  rset;
    int     sd,
            selsock,
            len,
            myv3server;
    u8      handshake_num,
            *buff;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)&peer_conn, sizeof(struct sockaddr_in))
      < 0) std_err();

    buff = malloc(BUFFSZ);
    if(!buff) std_err();

    if(fdloga) {
        memset(&tcp_info, 0, sizeof(tcp_info));
        tcp_info.sip  = get_peer_ip_port(sock, &tcp_info.sport);
        tcp_info.dip  = get_peer_ip_port(sd,   &tcp_info.dport);

        acp_dump_handshake(
            fdloga, SOCK_STREAM, IPPROTO_TCP,
            tcp_info.sip, tcp_info.sport, tcp_info.dip, tcp_info.dport,
            NULL, 0,
            &tcp_info.seq1, &tcp_info.ack1, &tcp_info.seq2, &tcp_info.ack2);
        fflush(fdloga);
    }

    len = proxy_ventrilo(sock, sd, buff);
    if(len < 0) goto disconnect;
    ventrilo_first_dec(buff, len);
    show_it("CLIENT", buff, len, &tcp_info);

    len = proxy_ventrilo(sd, sock, buff);
    if(len < 12) goto disconnect;
    ventrilo_first_dec(buff, len);
    show_it("SERVER", buff, len, &tcp_info);

    if(buff[8] != 4) {
        fprintf(fdlog, "- connection error: %s\n", buff + 12);
        goto disconnect;
    }
    if(ventrilo_read_keys(&client, &server, buff + 12, len - 12) < 0) {
        fputs("- no keys found, this is probably a new protocol\n", fdlog);
        goto disconnect;
    }
    myv3server = v3server;
    if(myv3server) handshake_num = 0;

    selsock = ((sd > sock) ? sd : sock) + 1;

    for(;;) {
        FD_ZERO(&rset);
        FD_SET(sock, &rset);
        FD_SET(sd, &rset);
        if(select(selsock, &rset, NULL, NULL, NULL)
          < 0) std_err();

        if(FD_ISSET(sd, &rset)) {            // from server
            len = proxy_ventrilo(sd, sock, buff);
            if(len < 0) break;
            ventrilo_dec(&server, buff, len);
            show_it("SERVER", buff, len, &tcp_info);
            if((myv3server == 2) && (buff[0] == 0x34)) {
                //memcpy(handshake_key, ventrilo3_auth[handshake_num].handshake_key, 64);
                ventrilo3_algo_scramble(&client, ventrilo3_auth[handshake_num].handshake_key);
                ventrilo3_algo_scramble(&server, ventrilo3_auth[handshake_num].handshake_key);
                myv3server++;
            }

        } else if(FD_ISSET(sock, &rset)) {    // from client
            len = proxy_ventrilo(sock, sd, buff);
            if(len < 0) break;
            ventrilo_dec(&client, buff, len);
            show_it("CLIENT", buff, len, &tcp_info);
            if((myv3server == 1) && (buff[0] == 72)) {
                handshake_num = buff[22];
                if(handshake_num > 3) break;
                if(!ventrilo3_auth[handshake_num].ok) {
                    printf("\nError: the handshake from the server %d has not received before\n", handshake_num);
                    break;
                }
                myv3server++;
            }
        }
    }

disconnect:
    close(sock);
    close(sd);
    free(buff);
    fputs("- disconnected\n", fdlog);
    return(0);
}



int proxy_ventrilo(int in, int out, u8 *buff) {
    int     t;
    u16     i,
            len;

    if(recv(in,  (u8 *)&len,     1, 0) <= 0) return(-1);
    if(recv(in,  (u8 *)&len + 1, 1, 0) <= 0) return(-1);
    if(send(out, (u8 *)&len,     2, 0) != 2) return(-1);
    len = ntohs(len);
    if(!len) return(len);

    for(i = 0; i < len; i += t) {
        t = recv(in, buff + i, len - i, 0);
        if(t <= 0) return(-1);
    }
    if(send(out, buff, len, 0) != len) return(-1);
    return(len);
}



void show_it(u8 *str, u8 *buff, int len, tcp_info_t *tcp_info) {
    u8      tmp[2];

    fprintf(fdlog, "    ### %s ###\n", str);
    show_dump(buff, len, fdlog);
    fflush(fdlog);

    if(fdloga) {
        tmp[0] = len >> 8;
        tmp[1] = len;

        if(str[0] == 'C') { // client
            /* acp_dump(    // the 2 bytes of the length
                fdloga, SOCK_STREAM, IPPROTO_TCP,
                tcp_info->sip, tcp_info->sport, tcp_info->dip, tcp_info->dport,
                tmp, 2,
                &tcp_info->seq1, &tcp_info->ack1, &tcp_info->seq2, &tcp_info->ack2); */

            acp_dump(
                fdloga, SOCK_STREAM, IPPROTO_TCP,
                tcp_info->sip, tcp_info->sport, tcp_info->dip, tcp_info->dport,
                buff, len,
                &tcp_info->seq1, &tcp_info->ack1, &tcp_info->seq2, &tcp_info->ack2);
        } else {            // server
            /* acp_dump(    // the 2 bytes of the length
                fdloga, SOCK_STREAM, IPPROTO_TCP,
                tcp_info->dip, tcp_info->dport, tcp_info->sip, tcp_info->sport,
                tmp, 2,
                &tcp_info->seq2, &tcp_info->ack2, &tcp_info->seq1, &tcp_info->ack1); */

            acp_dump(
                fdloga, SOCK_STREAM, IPPROTO_TCP,
                tcp_info->dip, tcp_info->dport, tcp_info->sip, tcp_info->sport,
                buff, len,
                &tcp_info->seq2, &tcp_info->ack2, &tcp_info->seq1, &tcp_info->ack1);
        }
        fflush(fdloga);
    }
}



u32 get_peer_ip_port(int sd, u16 *port) {
    struct sockaddr_in  peer;
    int     psz;

    psz = sizeof(struct sockaddr_in);
    if(getpeername(sd, (struct sockaddr *)&peer, &psz) < 0) {
        peer.sin_addr.s_addr = 0;
        peer.sin_port        = 0;
    }
    if(port) *port = peer.sin_port;
    return(peer.sin_addr.s_addr);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


