/*
    Copyright 2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ventrilo_udp.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define strnicmp    strncasecmp
#endif



#define VER         "0.1"
#define PORT        3784
#define MAXPCK      32
#define MAXPCKSZ    492
#define TIMEOUT     2
#define RETRY       3



void ventrilo_get_status(int sd, u_short cmd, u_char *pass);
int timeout(int sock);
u_int resolv(char *host);
void std_err(void);



struct  sockaddr_in peer;



int main(int argc, char *argv[]) {
    int     sd,
            i,
            cmd = -1,
            commands[] = {
                1,  // generic info
                2,  // details
                7,  // info visualized differently
                -1  // no other commands
            };
    u_short port = PORT;
    u_char  *pass = "",
            *host,
            *p,
            *l;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif


    setbuf(stdout, NULL);

    fputs("\n"
        "Ventrilo status retriever "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <host_or_url>\n"
            "\n"
            "Options:\n"
            "-p PORT   server port (default %hu)\n"
            "-c CMD    specify the command to use (default all)\n"
            "-w PASS   specify the server password\n"
            "\n"
            "Accepts also hosts containing port and password or urls like:\n"
            " www.somewhere.com:3784:mypass or ventrilo://1.2.3.4:1234/serverpassword=pass\n"
            "\n", argv[0], port);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 'p': port = atoi(argv[++i]); break;
            case 'c': cmd  = atoi(argv[++i]); break;
            case 'w': pass = argv[++i]; break;
            default: {
                printf("\nError: wrong cmd-line argument (%s)\n\n", argv[i]);
                exit(1);
            }
        }
    }

    host = argv[argc];
    p = strstr(host, "://");
    if(p) host = p + 3;

    p = strchr(host, ':');
    if(p) {
        *p++ = 0;
        port = atoi(p);
    } else {
        p = host;
    }

    if((l = strchr(p, ':'))) {
        pass = l + 1;

    } else if((l = strchr(p, '/'))) {
        *l = 0;
        do {
            p = l + 1;
            l = strchr(p, '=');
            if(!l) break;
            *l = 0;
            if(!strnicmp(p, "serverpassword", l - p)) pass = l + 1;
            p = l + 1;
        } while((l = strchr(p, '&')));
    }

    peer.sin_addr.s_addr = resolv(host);
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    printf("- target   %s : %hu\n",
        inet_ntoa(peer.sin_addr), port);

    if(*pass) printf("- Password %s\n", pass);

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    if(cmd < 0) {
        for(cmd = 0; commands[cmd] >= 0; cmd++) {
            ventrilo_get_status(sd, commands[cmd], pass);
        }
    } else {
        ventrilo_get_status(sd, cmd, pass);
    }

    close(sd);
    return(0);
}



void ventrilo_get_status(int sd, u_short cmd, u_char *pass) {
    ventrilo_udp_head   *stat;
    int     i,
            len,
            totlen,
            retry;
    u_short id,
            crc;
    u_char  buff[20 + MAXPCKSZ],
            full[MAXPCKSZ * MAXPCK],
            *data;

    stat = (ventrilo_udp_head *)buff;
    data = buff + 20;

    printf("\n- Command code %d:\n\n", cmd);

    strncpy(data, pass, 16);

    stat->zero    = 0;
    stat->cmd     = cmd;
    stat->id      = id = time(NULL);
    stat->totlen  = 16;
    stat->len     = 16;
    stat->totpck  = 1;
    stat->pck     = 0;
    stat->crc     = ventrilo_udp_crc(data, 16);
    stat->datakey = ventrilo_udp_data_enc(data, 16);
    ventrilo_udp_head_enc(buff);

    for(retry = RETRY; retry; retry--) {
        sendto(sd, buff, 20 + 16, 0, (struct sockaddr *)&peer, sizeof(peer));
        if(!timeout(sd)) break;
    }
    if(!retry) {
        fputs("\nError: no reply received, probably the server is not online\n\n", stdout);
        exit(1);
    }

    i      = 0;
    totlen = 0;
    memset(full, ' ', sizeof(full));    // in case of packet loss

    for(;;) {
        len = recvfrom(sd, buff, sizeof(buff), 0, NULL, NULL);
        ventrilo_udp_head_dec(buff);

        if(stat->id != id) continue;

        if((len < 20)                 ||
           (stat->totpck < stat->pck) ||
           (stat->totpck > MAXPCK)    ||
           (stat->len    > MAXPCKSZ)) {
            fputs("\nError: wrong or incomplete reply received\n", stdout);
            return;
        }

        len    = stat->len;
        totlen += len;
        if(totlen > sizeof(full)) break;

        ventrilo_udp_data_dec(data, len, stat->datakey);
        memcpy(full + (stat->pck * MAXPCKSZ), data, len);

        if(++i == stat->totpck) break;
        if(totlen == stat->totlen) break;
        if(timeout(sd) < 0) break;
    }

    crc = ventrilo_udp_crc(full, totlen);
    if(ventrilo_udp_crc(full, totlen) != stat->crc) {
        printf("- wrong checksum: mine is %04x while should be %04x\n\n", crc, stat->crc);
    }

    fwrite(full, totlen, 1, stdout);
}



int timeout(int sock) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec = TIMEOUT;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



u_int resolv(char *host) {
    struct hostent *hp;
    u_int  host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u_int *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif

