/*
    Copyright 2007 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <ctype.h>

#ifdef WIN32
    #include <direct.h>

    #define u_int   unsigned int
    #define u_short unsigned short
    #define u_char  unsigned char
    #define PATHSLASH   '\\'
#else
    #include <unistd.h>

    #define PATHSLASH   '/'
#endif



#define VER     "0.1"
#define BUFFSZ  8192



int vital_dec(int filesize, u_char *data, int size);
int fd16(FILE *fd);
int fd32(FILE *fd);
int get16(u_char *data, int *num);
int get32(u_char *data, int *num);
u_char *create_dir(u_char *name);
void std_err(void);



int main(int argc, char *argv[]) {
    struct stat xstat;
    FILE    *fd,
            *fdo;
    u_int   vkey,
            len,
            filever,
            files,
            filesize,
            files_num,
            names_len,
            offset;
    u_char  buff[BUFFSZ],
            *in_file,
            *out_dir,
            *out_file,
            *names,
            *p;

    setbuf(stdout, NULL);

    fputs("\n"
        "Vital engine files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <file.GRP> <output_folder>\n"
            "\n", argv[0]);
        exit(1);
    }

    in_file = argv[1];
    out_dir = argv[2];

    printf("- open file %s\n", in_file);
    fd = fopen(in_file, "rb");
    if(!fd) std_err();

    printf("- change folder %s\n", out_dir);
    if(chdir(out_dir) < 0) std_err();

    fread(buff, 4, 1, fd);
    if(memcmp(buff, "FPRG", 4)) {
        printf("\nError: this is not a FPRG file\n");
        exit(1);
    }
    filever   = fd32(fd);
    files_num = fd32(fd);
    names_len = fd32(fd);
    fstat(fileno(fd), &xstat);

    names = malloc(names_len);
    if(!names) std_err();
    fread(names, names_len, 1, fd);
    vkey = vital_dec(xstat.st_size, names, names_len);
    if(vkey < 0) {
        printf("\n"
            "Error: the decryption algorithm resulted in a too big filename size\n"
            "       Please contact me reporting this problem\n");
        exit(1);
    }
    printf("- key number 0x%x\n", vkey);

    printf("\n"
        "  offset   filesize   filename\n"
        "  ----------------------------\n");

    out_file = buff;
    p = names;
    for(files = 0; files < files_num; files++) {
        p += get16(p, &len);                // length of the filename
        if(len > 260) {
            printf("\nError: wrong filename size (%u) probably the decryption is wrong\n", len);
            exit(1);
        }
        sprintf(out_file, "%.*s", len, p);  // filename
        p += len;
        p += get32(p, &offset);             // offset
        p += get32(p, &filesize);           // filesize
        p += get32(p, NULL);                // attributes?

        printf("  %08x %-10u %s\n", offset, filesize, out_file);

        fdo = fopen(create_dir(out_file), "wb");
        if(!fdo) std_err();

        for(len = sizeof(buff); filesize; filesize -= len) {
            if(filesize < len) len = filesize;
            fread(buff,  len, 1, fd);
            fwrite(buff, len, 1, fdo);
        }

        fclose(fdo);
    }

    free(names);
    fclose(fd);
    printf("\n- %u files extracted\n", files);
    return(0);
}



int vital_dec(int filesize, u_char *data, int size) {
    u_short size16;
    int     i,
            key,
            keyadd;
    u_char  *limit;

    keyadd = 0x535a;

        /* unfortunately Codename Outbreak demo uses 0x535a */
        /* while Boiling Point uses 0x737a so I launch this */
        /* simple scanner which verifies if data[1] is zero */
        /* since a filename is usually smaller than 260.    */
        /* the filever is useless since it's 1 for both.    */

        /* scanning */
    for(i = 4; i; i--) {
        key = (filesize + keyadd) % 0xffff;
        key = (key * 0x343fd) + 0x269ec3;   // data[0]
        size16 = data[0] ^ (((key >> 16) & 0x7fff) % 0xff);
        key = (key * 0x343fd) + 0x269ec3;   // data[1]
        size16 |= (data[1] ^ (((key >> 16) & 0x7fff) % 0xff)) << 8;
        if(size16 <= 260) break;
        keyadd += 0x2000;
    }
    if(!i) return(-1);

        /* decoding */
    key = (filesize + keyadd) % 0xffff;
    for(limit = data + size; data != limit; data++) {
        key = (key * 0x343fd) + 0x269ec3;
        *data ^= ((key >> 16) & 0x7fff) % 0xff;
    }

    return(keyadd);
}



int fd16(FILE *fd) {
    return(fgetc(fd) | (fgetc(fd) << 8));
}



int fd32(FILE *fd) {
    return(fgetc(fd) | (fgetc(fd) << 8) | (fgetc(fd) << 16) | (fgetc(fd) << 24));
}



int get16(u_char *data, int *num) {
    if(num) *num = data[0] | (data[1] << 8);
    return(2);
}



int get32(u_char *data, int *num) {
    if(num) *num = data[0] | (data[1] << 8) | (data[2] << 16) | (data[3] << 24);
    return(4);
}



u_char *create_dir(u_char *name) {
    u_char  *p,
            *l;

    for(p = name; (*p == '\\') || (*p == '/') || (*p == '.'); p++);
    name = p;

    for(;;) {
        if((p[0] && (p[1] == ':')) || ((p[0] == '.') && (p[1] == '.'))) p[0] = '_';

        l = strchr(p, '\\');
        if(!l) {
            l = strchr(p, '/');
            if(!l) break;
        }
        *l = 0;
        p = l + 1;

#ifdef WIN32
        mkdir(name);
#else
        mkdir(name, 0755);
#endif
        *l = PATHSLASH;
    }
    return(name);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


