/*
    Copyright 2007 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www->gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include "acpdump.h"

#ifdef WIN32
#else
    #define stricmp strcasecmp
    #define stristr strcasestr
#endif



#define VER         "0.2"
#define BUFFSZ      0xffff
#define htons       net16
#define ntohs       net16
#define htonl       net32
#define ntohl       net32
#define inet_addr   str2ip
#define inet_ntoa   ip2str

#define GETTYPE(x)  type = (stristr(x, "recvfrom") || stristr(x, "sendto")) ? 17 : 6;



void dump(FILE *fdo, int type, uint32_t sip, uint16_t sport, uint32_t dip, uint16_t dport, uint8_t *data, int datalen);
int h2b16(uint8_t *in, uint8_t *out);
void get_ip_port(uint8_t *data, uint32_t *ip, uint16_t *port);
uint8_t *skip_space(uint8_t *data);
uint8_t *fgetz(FILE *fd, uint8_t *data);
uint32_t get32(FILE *fd);
uint8_t *getmm(FILE *fd, uint8_t *data, uint32_t len);
void std_err(void);



#pragma pack(2)
typedef struct {
    uint32_t    sip;
    uint16_t    sport;
    uint32_t    dip;
    uint16_t    dport;
    uint8_t     type;
    uint32_t    seq1;
    uint32_t    ack1;
    uint32_t    seq2;
    uint32_t    ack2;
} wpe_hosts_t;
#pragma pack()

wpe_hosts_t *wpe_hosts;
uint32_t    pck;
int         wpe_hosts_memsize,
            wpe_hosts_len;



int main(int argc, char *argv[]) {
    FILE        *fd,
                *fdo;
    uint32_t    num,
                datalen,
                src_ip,
                dst_ip;
    uint16_t    src_port,
                dst_port;
    int         pac     = -1,
                type;
    uint8_t     *data,
                *tmp,
                *in_file,
                *out_file,
                *p;

    setbuf(stdout, NULL);

    fputs("\n"
        "WPE packet format to tcpdump capture format "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <input.PAC/TXT> <output.CAP>\n"
            "\n", argv[0]);
        exit(1);
    }

    in_file  = argv[1];
    out_file = argv[2];

    p = strrchr(in_file, '.');
    if(p) {
        if(!stricmp(p, ".pac")) pac = 1;
        if(!stricmp(p, ".txt")) pac = 0;
    }
    if(pac < 0) {
        printf("\nError: the input file doesn't thave the PAC or TXT extension\n");
        exit(1);
    }

    printf("- open input file %s\n", in_file);
    fd = fopen(in_file, "rb");
    if(!fd) std_err();

    printf("- create output file %s\n", out_file);
    fdo = fopen(out_file, "wb");
    if(!fdo) std_err();
    create_acp(fdo);

    wpe_hosts_len     = 0;
    wpe_hosts_memsize = 32;
    data = malloc(BUFFSZ + 1);
    tmp  = malloc(BUFFSZ + 1);
    wpe_hosts = malloc(wpe_hosts_memsize * sizeof(wpe_hosts_t));
    if(!data || !tmp || !wpe_hosts) std_err();

    if(pac) {
        printf("- PAC file format\n");

        num = get32(fd);

        for(pck = 0; pck < num; pck++) {
            datalen = get32(fd);            // data
            getmm(fd, data, datalen);

            get32(fd);                      // ???

            getmm(fd, tmp, get32(fd));      // source IP
            get_ip_port(tmp, &src_ip, &src_port);

            getmm(fd, tmp, get32(fd));      // dest IP
            get_ip_port(tmp, &dst_ip, &dst_port);

            getmm(fd, tmp, get32(fd));      // data size (useless)

            getmm(fd, tmp, get32(fd));      // function
            GETTYPE(tmp)

            dump(fdo, type, src_ip, src_port, dst_ip, dst_port, data, datalen);
        }

    } else {
        printf("- TXT file format\n");

        while(fgetz(fd, tmp)) {
            if(!tmp[0]) continue;
                                                p = tmp;
            pck = atoi(p);                      p = skip_space(p);
            get_ip_port(p, &src_ip, &src_port); p = skip_space(p);
            get_ip_port(p, &dst_ip, &dst_port); p = skip_space(p);
            datalen = atoi(p);                  p = skip_space(p);
            GETTYPE(p)

            datalen = 0;
            while(fgetz(fd, tmp)) {
                if(!tmp[0]) break;

                datalen += h2b16(skip_space(tmp), data + datalen);
                if((datalen + 16) > BUFFSZ) break;
            }

            dump(fdo, type, src_ip, src_port, dst_ip, dst_port, data, datalen);
        }
    }

    printf("- finished\n");
    fclose(fd);
    fclose(fdo);
    free(data);
    free(tmp);
    free(wpe_hosts);
    return(0);
}



void dump(FILE *fdo, int type, uint32_t sip, uint16_t sport, uint32_t dip, uint16_t dport, uint8_t *data, int datalen) {
    wpe_hosts_t *w;
    int         i;

    printf("  %s %-10u %u bytes from %s:%hu to ",
        (type == 17) ? "UDP" : "TCP",           // show info
        pck,
        datalen,
        inet_ntoa(sip), ntohs(sport));
    printf("%s:%hu\n",
        inet_ntoa(dip), ntohs(dport));

    if(type != 6) {                             // if not tcp (recvfrom/sendto)
        acp_dump(
            fdo, type,
            sip, sport, dip, dport,
            data, datalen,
            NULL, NULL, NULL, NULL);
        return;
    }

    for(i = 0; i < wpe_hosts_len; i++) {        // if already in the database
        w = &wpe_hosts[i];

        if(
          (sip   == w->sip)   &&
          (sport == w->sport) &&
          (dip   == w->dip)   &&
          (dport == w->dport) &&
          (type  == w->type)) {
            acp_dump(
                fdo, type,
                sip, sport, dip, dport,
                data, datalen,
                &w->seq1, &w->ack1, &w->seq2, &w->ack2);
            return;
        }
    }

    if(wpe_hosts_len >= wpe_hosts_memsize) {    // if no space
        wpe_hosts_memsize += 32;
        wpe_hosts = realloc(wpe_hosts, wpe_hosts_memsize * sizeof(wpe_hosts_t));
        if(!wpe_hosts) std_err();
    }

    printf("- new TCP connection\n");           // add new entry
    w = &wpe_hosts[i];

    w->sip      = sip;
    w->sport    = sport;
    w->dip      = dip;
    w->dport    = dport;
    w->type     = type;
    w->seq1     = 0;
    w->ack1     = 0;
    w->seq2     = 0;
    w->ack2     = 0;
    wpe_hosts_len++;

    acp_dump_handshake(
        fdo, type,
        sip, sport, dip, dport,
        NULL, 0,
        &w->seq1, &w->ack1, &w->seq2, &w->ack2);

    acp_dump(
        fdo, type,
        sip, sport, dip, dport,
        data, datalen,
        &w->seq1, &w->ack1, &w->seq2, &w->ack2);
}


int h2b16(uint8_t *in, uint8_t *out) {
    int         i,
                n;
    uint8_t     *o;

    o = out;
    for(i = 0; i < 16; i++) {
        if(in[0] <= ' ') break;
        sscanf(in, "%02x", &n);
        *o = n;
        in += 3;
        o++;
    }

    return(o - out);
}



void get_ip_port(uint8_t *data, uint32_t *ip, uint16_t *port) {
    uint8_t     *p;

    p = strchr(data, ':');
    if(!p) {
        *ip   = 0;
        *port = 0;
        return;
    }
    *p = 0;
    *ip   = inet_addr(data);
    *port = htons(atoi(p + 1));
    *p = ':';
}



uint8_t *skip_space(uint8_t *data) {
    uint8_t     *p;

    for(p = data; *p &&  (*p != ' ') && (*p != '\t');  p++);
    for(        ; *p && ((*p == ' ') || (*p == '\t')); p++);
    return(p);
}



uint8_t *fgetz(FILE *fd, uint8_t *data) {
    uint8_t     *p;

    if(!fgets(data, BUFFSZ, fd)) return(NULL);

    for(p = data; *p && (*p != '\n') && (*p != '\r'); p++);
    *p = 0;

    return(data);
}



uint32_t get32(FILE *fd) {
    return(fgetc(fd) | (fgetc(fd) << 8) | (fgetc(fd) << 16) | (fgetc(fd) << 24));
}



uint8_t *getmm(FILE *fd, uint8_t *data, uint32_t len) {
    if(len > BUFFSZ) {
        fread(data, BUFFSZ, 1, fd);
        data[BUFFSZ] = 0;
        fseek(fd, len - BUFFSZ, SEEK_CUR);
    } else {
        fread(data, len, 1, fd);
        data[len] = 0;
    }
    return(data);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


