/*
    Copyright 2008,2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

/*
  thanx to Diego "doppiapunta" Defilippi of http://www.figlidigaucci.eu/ for ideas/suggestions/testing!
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>

#ifdef WIN32
    #include <direct.h>
    #include <windows.h>
    #define PATHSLASH   '\\'
#else
    #define stricmp     strcasecmp
    #define PATHSLASH   '/'
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1.1b"
#define PATHMAX     4096



u8 *get_line(u8 **line);
void txt2xso(u8 *data, int datasz);
void check_overwrite(u8 *fname);
int getxx(u8 *data, u32 *ret, int bits);
void xso2txt(FILE *fd, u8 *fname);
int recursive_dir(u8 *filedir, FILE *fd);
void create_dir(u8 *name);
void myalloc(u8 **data, unsigned wantsize, unsigned *currsize);
u8 *fdload(u8 *fname, int *fsize);
void std_err(void);



int     files_done  = 0;



int main(int argc, char *argv[]) {
    FILE    *fd,
            *fd_check;
    int     datasz;
    u8      filedir[PATHMAX + 1],
            *oper,
            *fdir,
            *fname,
            *data,
            *p;

    setbuf(stdout, NULL);

    fputs("\n"
        "Falcom YS games XSO files extractor and rebuilder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 4) {
        printf("\n"
            "Usage: %s <operation> <folder> <file.txt>\n"
            "\n"
            "Operations:\n"
            " e = extract\n"
            " r = rebuild\n"
            "\n"
            "When you specify the \"extract\" operation the field \"folder\" is the input data\n"
            "folder (for example the one obtained through my ysext tool for extracting the\n"
            "NA/NI archives) and \"file.txt\" is the output text file generated by this tool\n"
            "which will contain all the phrases and informations retrieved from the original\n"
            "XSO file, examples:\n"
            "  xsoext e c:\\ysf_win\\extracted_data_1101 xso.txt\n"
            "  xsoext e c:\\ysf_win\\extracted_data_1101\\MAP\\S_05\\S_0551\\S_0551.XSO xso.txt\n"
            "\n"
            "instead the \"rebuild\" operation inverts the process, so the tool will read\n"
            "\"file.txt\" as input and will generate all the XSO files using the informations\n"
            "contained in it, example: xsoext r c:\\ysf_win\\extracted_data_1101 xso.txt\n"
            "\n"
            "How to open and read \"file.txt\" on non-japanese systems:\n"
            "- download AppLocale from http://www.microsoft.com/globaldev/tools/apploc.mspx\n"
            "- launch applocale, add notepad.exe and select the last language in the list\n"
            "- open file.txt and you will be able to see all the kanji chars\n"
            "\n", argv[0]);
        exit(1);
    }

    oper  = argv[1];
    fdir  = argv[2];
    fname = argv[3];

    while(*oper && ((*oper == '-') || (*oper == '/'))) oper++;
    if(tolower(oper[0]) == 'e') {
        printf("- create file %s\n", fname);
        check_overwrite(fname);
        fd = fopen(fname, "wb");
        if(!fd) std_err();
        fprintf(fd,
            "  // use applocale + a text editor to modify this file:\n"
            "  //   http://www.microsoft.com/globaldev/tools/apploc.mspx\n"
            "  // do NOT use the Windows notepad since it's bugged!\n");

        fd_check = fopen(fdir, "rb");
        if(fd_check) {
            fclose(fd_check);
            p = strrchr(fdir, '\\');
            if(!p) p = strrchr(fdir, '/');
            if(p) {
                *p = 0;
                if(chdir(fdir) < 0) std_err();
                fdir = p + 1;
            }
            xso2txt(fd, fdir);
        } else {
            printf("- set input folder %s\n", fdir);
            if(chdir(fdir) < 0) std_err();
            printf("- start the scanning of the input folder for XSO files\n");
            strcpy(filedir, ".");
            recursive_dir(filedir, fd);
        }
        fclose(fd);

    } else if(tolower(oper[0]) == 'r') {
        printf("- open file");
        data = fdload(fname, &datasz);

        printf("- set output folder %s\n", fdir);
        if(chdir(fdir) < 0) std_err();

        txt2xso(data, datasz);

        free(data);

    } else {
        printf("\nError: wrong operation, use 'e' for extraction or 'r' for rebuilding\n");
        exit(1);
    }

    printf("- %d files handled\n", files_done);
    return(0);
}



u8 *get_line(u8 **line) {
    int     i,
            crlf;
    u8      *p;
    static int  buffsz = 0;
    static u8   *buff  = NULL;

    i = 0;
    crlf = 0;
    for(p = *line; *p; p++) {
        if(i >= buffsz) {
            buffsz += 10237;    // memory trick
            buff = realloc(buff, buffsz);
            if(!buff) std_err();
        }
        if((*p == '\n') || (*p == '\r')) {
            if(!crlf) {
                buff[i++] = 0;
                crlf = 1;
            }
            continue;
        }
        if(crlf) {
            while(*p && ((*p == '\n') || (*p == '\r'))) p++;
            break;
        }
        buff[i++] = *p;
    }
    if(!*p) buff[i++] = 0;
    *line = p;
    return(buff);
}



void num2xso(FILE *fd, u8 *data) {
    u32     num;
    int     n;
    u8      *p;

    p = data;
    while(*p) {
        n = -1;
        if(sscanf(p, "%i%n", &num, &n) != 1) break;
        if(n < 0) break;
        fwrite(&num, 1, sizeof(num), fd);
        for(p += n; *p && ((*p == ' ') || (*p == '\t')); p++);
    }
}



void dump_phrases(FILE *fd, u8 **phrase, int max) {
    u32     off;
    int     i;

    off = 0;
    for(i = 0; i < max; i++) {
        fwrite(&off, 1, sizeof(off), fd);
        off += strlen(phrase[i]) + 1;
    }

    for(i = 0; i < max; i++) {
        fwrite(phrase[i], 1, strlen(phrase[i]) + 1, fd);
        free(phrase[i]);
    }

    for(off += max * 4; off & 3; off++) fputc(0x00, fd);
}



void txt2xso(u8 *data, int datasz) {
    FILE    *fd;
    int     phrasenum,
            phrasesz;
    u8      *line,
            *limit,
            *txt,
            *val,
            **phrase;

    fd = NULL;
    phrase = NULL;
    phrasenum = 0;
    phrasesz = 0;
    line = data;
    limit = data + datasz;

    while(line < limit) {
        txt = get_line(&line);
        if(!txt[0]) continue;
        while(*txt && ((*txt == ' ') || (*txt == '\t'))) txt++;
        if((txt[0] == '#') || (txt[0] == ';') || (txt[0] == '/')) continue; // comments

        val = strchr(txt, ':');
        if(!val) continue;
        *val++ = 0;
        // for(*val++ = 0; *val; val++) *val = 0;

        if(!stricmp(txt, "file")) {
            printf("  %s\n", val);
            if(fd) {
                dump_phrases(fd, phrase, phrasenum);
                phrasenum = 0;
                fclose(fd);
            }
            //check_overwrite(val);
            create_dir(val);
            fd = fopen(val, "wb");
            if(!fd) std_err();
            fwrite("XSR\0", 1, 4, fd);
            files_done++;
        } else if(!stricmp(txt, "header")) {
            if(!fd) continue;
            num2xso(fd, val);
        } else if(!stricmp(txt, "data")) {
            if(!fd) continue;
            num2xso(fd, val);
        } else {
            if(!fd) continue;
            if(phrasenum >= phrasesz) {
                phrasesz += 1024;
                phrase = realloc(phrase, sizeof(u8 *) * phrasesz);
                if(!phrase) std_err();
            }
            phrase[phrasenum++] = strdup(val);
        }
    }
    if(fd) {
        dump_phrases(fd, phrase, phrasenum);
        phrasenum = 0;
        fclose(fd);
    }
}



void check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return;
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') exit(1);
}



int getxx(u8 *data, u32 *ret, int bits) {
    u32     num;
    int     i,
            bytes;

    bytes = bits >> 3;
    for(num = i = 0; i < bytes; i++) {
         num |= (data[i] << (i << 3));
    }
    *ret = num;
    return(bytes);
}



void xso2txt(FILE *fd, u8 *fname) {
    u32     num,
            tot1,
            tot2;
    int     i,
            datasz;
    u8      *data,
            *p,
            *text,
            *limit;

    p = strrchr(fname, '.');
    if(!p || stricmp(p, ".xso")) return;

    data = fdload(fname, &datasz);
    if(datasz < 0x24) goto quit;
    if(memcmp(data, "XSR\0", 4)) goto quit;

    files_done++;
    fprintf(fd, "file:%s\r\n", fname);
    p = data + 4;
    limit = data + datasz;
    // the files come from the official game so I have no implemented additional checks or security checks!!!

    fprintf(fd, "header:");
    for(i = 0; i < 6; i++) {
        p += getxx(p, &num, 32);
        fprintf(fd, "%i ", num);
    }
    p += getxx(p, &tot1, 32);
    fprintf(fd, "%i ", tot1);
    p += getxx(p, &tot2, 32);
    fprintf(fd, "%i ", tot2);
    fprintf(fd, "\r\n");

    text = p + (tot1 * 4) + (tot2 * 4);
    if(text > limit) goto quit;

    fprintf(fd, "data:");
    for(i = 0; i < tot1; i++) {
        p += getxx(p, &num, 32);
        fprintf(fd, "%i ", num);
    }
    fprintf(fd, "\r\n");

    for(i = 0; i < tot2; i++) {
        p += getxx(p, &num, 32);
        fprintf(fd, "%d:%s\r\n", i, text + num);
    }
    fprintf(fd, "\r\n");

quit:
    free(data);
}



int recursive_dir(u8 *filedir, FILE *fd) {
    int     plen,
            ret     = -1;

#ifdef WIN32
    static int      winnt = -1;
    OSVERSIONINFO   osver;
    WIN32_FIND_DATA wfd;
    HANDLE  hFind;

    if(winnt < 0) {
        osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        GetVersionEx(&osver);
        if(osver.dwPlatformId >= VER_PLATFORM_WIN32_NT) {
            winnt = 1;
        } else {
            winnt = 0;
        }
    }

    plen = strlen(filedir);
    strcpy(filedir + plen, "\\*.*");
    plen++;

    if(winnt) { // required to avoid problems with Vista and Windows7!
        hFind = FindFirstFileEx(filedir, FindExInfoStandard, &wfd, FindExSearchNameMatch, NULL, 0);
    } else {
        hFind = FindFirstFile(filedir, &wfd);
    }
    if(hFind == INVALID_HANDLE_VALUE) return(0);
    do {
        if(!strcmp(wfd.cFileName, ".") || !strcmp(wfd.cFileName, "..")) continue;

        strcpy(filedir + plen, wfd.cFileName);

        if(wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
            if(recursive_dir(filedir, fd) < 0) goto quit;
        } else {
            xso2txt(fd, filedir + 2);
        }
    } while(FindNextFile(hFind, &wfd));
    ret = 0;

quit:
    FindClose(hFind);
#else
    struct  stat    xstat;
    struct  dirent  **namelist;
    int     n,
            i;

    n = scandir(filedir, &namelist, NULL, NULL);
    if(n < 0) {
        if(stat(filedir, &xstat) < 0) {
            printf("**** %s", filedir);
            std_err();
        }
        xso2txt(fd, filedir + 2);
        return(0);
    }

    plen = strlen(filedir);
    strcpy(filedir + plen, "/");
    plen++;

    for(i = 0; i < n; i++) {
        if(!strcmp(namelist[i]->d_name, ".") || !strcmp(namelist[i]->d_name, "..")) continue;
        strcpy(filedir + plen, namelist[i]->d_name);

        if(stat(filedir, &xstat) < 0) {
            printf("**** %s", filedir);
            std_err();
        }
        if(S_ISDIR(xstat.st_mode)) {
            if(recursive_dir(filedir, fd) < 0) goto quit;
        } else {
            xso2txt(fd, filedir + 2);
        }
        free(namelist[i]);
    }
    ret = 0;

quit:
    for(; i < n; i++) free(namelist[i]);
    free(namelist);
#endif
    filedir[plen - 1] = 0;
    return(ret);
}



void create_dir(u8 *name) {
    u8      *p,
            *l;

        // removed any security check since the files are ever officials

    for(p = name;; p = l + 1) {
        l = strchr(p, '\\');
        if(!l) l = strchr(p, '/');
        if(!l) break;
        *l = 0;

#ifdef WIN32
        mkdir(name);
#else
        mkdir(name, 0755);
#endif
        *l = PATHSLASH;
    }
}



void myalloc(u8 **data, unsigned wantsize, unsigned *currsize) {
    if(wantsize <= *currsize) return;
    *data = realloc(*data, wantsize);
    if(!*data) std_err();
    *currsize = wantsize;
}



u8 *fdload(u8 *fname, int *fsize) {
    struct stat xstat;
    FILE    *fd;
    int     size;
    u8      *buff;

    printf("  %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();
    fstat(fileno(fd), &xstat);
    size = xstat.st_size;
    buff = malloc(size);
    if(!buff) std_err();
    fread(buff, 1, size, fd);
    fclose(fd);
    *fsize = size;
    return(buff);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


