/*
    Copyright 2008,2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

/*
  thanx to Diego "doppiapunta" Defilippi of http://www.figlidigaucci.eu/ for ideas/suggestions/testing!
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>
#include <zlib.h>

#ifdef WIN32
    #include <windows.h>
    #define PATHSLASH   '\\'
#else
    #define stricmp     strcasecmp
    #define PATHSLASH   '/'
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1.3b"
#define EXTRACT     0
#define REBUILD     1
#define APPEND      2
#define NIDECRYPT   3
#define LISTONLY    4
#define COMPRESSZ   5
#define NIENCRYPT   6
#define ZLIBMAX(X)  ((X) + ((X) / 1000) + 12)

#define PATHMAX     4096



#pragma pack(4)
typedef struct {
    u32     sign;
    u32     tot;
    u32     namesz;
    u32     zero;
} nni_t;

typedef struct {
    u32     num;
    u32     size;
    u32     offset;
    u32     nameoff;
} nni_info_t;

typedef struct {
    u32     num;
    u8      *fname;
} num_list_t;
#pragma pack()



int check_wildcard(u8 *fname, u8 *wildcard);
u32 get_num_list(u8 *fname);
void build_num_list(u8 *numfile);
void read_ni(u8 *fname, u8 *ext, nni_t **nni_ret, nni_info_t **nni_info_ret, u8 **nni_names_ret, int crypt);
void check_overwrite(u8 *fname);
int add_files(u8 *fname, u32 fsize, nni_t *nni_in, nni_info_t *nni_info_in, u8 *nni_names_in);
void toupperz(u8 *in, u8 *out, int addz);
int recursive_dir(u8 *filedir);
void myalloc(u8 **data, unsigned wantsize, unsigned *currsize);
void dumpa(u8 *fname, u8 *data, int datasz);
void create_dir(u8 *name) ;
void myfr(FILE *fd, void *data, unsigned size);
void myfw(FILE *fd, void *data, unsigned size);
int zip(u8 *in, u32 insz, u8 *out, u32 outsz);
int unzip(u8 *in, u32 insz, u8 *out, u32 outsz);
void ys_ni_decrypt(u8 *data, int datasz);
void ys_ni_encrypt(u8 *data, int datasz);
void show_examples(void);
void show_info(void);
void std_err(void);



num_list_t  *num_list;
z_stream    z;
FILE        *fd_ni,
            *fd_na;
int         overwrite   = 0,
            operation   = EXTRACT;
u8          *force_wildcard  = NULL;



int main(int argc, char *argv[]) {
    nni_info_t  *nni_info;
    struct stat xstat;
    nni_t   *nni;
    FILE    *fd         = NULL,
            *fd_num     = NULL;
    u32     crc,
            size,
            zsize;
    int     i,
            insz,
            outsz,
            tot;
    u8      filedir[PATHMAX + 1],
            *fname,
            *fdir,
            *ext,
            *nni_names,
            *in,
            *out,
            *name,
            *numfile    = NULL;

    setbuf(stdout, NULL);

    fputs("\n"
        "Falcom YS games NA/NI/Z files extractor and rebuilder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc == 2) {
        if(!stricmp(argv[1], "-e")) show_examples();
        if(!stricmp(argv[1], "-i")) show_info();
    }
    if(argc < 3) {
        printf("\n"
            "Usage: %s [options] <file.NI/NA/Z> <folder>\n"
            "\n"
            "Options:\n"
            "-r     rebuild the NI/NA archive using folder as input folder containing all\n"
            "       the files you want to add\n"
            "-a     append the files contained in folder to a pre-existent NA archive while\n"
            "       the NI will be rebuilt since its format doesn't allow to append info\n"
            "-x     decrypt a NI file without extracting its content (debug)\n"
            "-X     encrypt a decrypted NI file (debug)\n"
            "-l     list the files without extracting them\n"
            "-n FD  specify the file which contains the number assigned to each archived\n"
            "       file. this option is VERY IMPORTANT if you want to rebuild the NA\n"
            "       archive since without the correct numbers the game will not work\n"
            "       FD is the name of the file, output file in extraction or input in -r/-a\n"
            "-f \"W\" extract/list/rebuild only the files which respects the W wildcard\n"
            "-o     overwrite any existent file in folder during extraction without asking\n"
            "-z     compress a file into a Z file\n"
            "-e     show some usage examples\n"
            "-i     show some informations about the NI, NA and Z files and their purpose\n"
            "\n"
            "the default operation is the extracting of the files to the output folder.\n"
            "folder is ignored with .Z files which will be created in the original location\n"
            "without the .Z extension (so c:\\file.dds.z will be recreated as c:\\file.dds)\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 2;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case '-':
            case '?':
            case 'h': {
                exit(1);
                } break;
            case 'r': {
                operation   = REBUILD;
                } break;
            case 'a': {
                operation   = APPEND;
                } break;
            case 'x': {
                operation   = NIDECRYPT;
                } break;
            case 'X': {
                operation   = NIENCRYPT;
                } break;
            case 'l': {
                operation   = LISTONLY;
                } break;
            case 'n': {
                if(!argv[++i]) exit(1);
                numfile     = argv[i];
                } break;
            case 'f': {
                if(!argv[++i]) exit(1);
                force_wildcard  = argv[i];
                } break;
            case 'o': {
                overwrite   = 1;
                } break;
            case 'z': {
                operation   = COMPRESSZ;
                } break;
            case 'e': {
                show_examples();
                } break;
            case 'i': {
                show_info();
                } break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }

    fname = malloc(strlen(argv[argc]) + 16);
    strcpy(fname, argv[argc]);
    fdir  = argv[argc + 1];

    z.zalloc = (alloc_func)0;
    z.zfree  = (free_func)0;
    z.opaque = (voidpf)0;
    tot      = 0;
    in       = out   = NULL;
    insz     = outsz = 0;

    if(operation == COMPRESSZ) {
        if(deflateInit2(&z, Z_BEST_COMPRESSION, Z_DEFLATED, 15, 9, Z_DEFAULT_STRATEGY) != Z_OK) {
            printf("\nError: zlib initialization error\n");
            exit(1);
        }

        printf("- open input file %s\n", fname);
        fd = fopen(fname, "rb");
        if(!fd) std_err();
        fstat(fileno(fd), &xstat);
        size = xstat.st_size;
        strcat(fname, ".z");
        name = fname;

        myalloc(&in,  size, &insz);
        myfr(fd, in, size);

        myalloc(&out, 8 + ZLIBMAX(size), &outsz);
        zsize = zip(in, size, out + 8, outsz - 8);

        crc = crc32(0, in, size);
        *(u32 *)(out)     = crc;
        *(u32 *)(out + 4) = size;

        dumpa(name, out, 8 + zsize);
        tot++;

        goto quit;
    }

    ext = strrchr(fname, '.');
    if(!ext || (stricmp(ext, ".ni") && stricmp(ext, ".na") && stricmp(ext, ".z"))) {
        printf("\nError: wrong input file extension, it must be NI, NA or Z\n");
        exit(1);
    }

    if(operation == APPEND) {
        read_ni(fname, ext, &nni, &nni_info, &nni_names, NIDECRYPT);
    }

    if((operation == REBUILD) || (operation == APPEND)) {
        printf("- REBUILD/APPEND mode\n");

        build_num_list(numfile);

        strcpy(ext, ".ni");
        printf("- create index file %s\n", fname);
        if(operation == APPEND) printf("  the NI file must be rebuilt in append mode too\n");
        check_overwrite(fname);
        fd_ni = fopen(fname, "wb");
        if(!fd_ni) std_err();

        strcpy(ext, ".na");
        if(operation == APPEND) {
            printf("- append data to file %s\n", fname);
            fd_na = fopen(fname, "r+b");
            if(!fd_na) std_err();
            fflush(fd_na);
            fseek(fd_na, 0, SEEK_END);
            add_files(NULL, 0, nni, nni_info, nni_names);
        } else {
            printf("- create data file %s\n", fname);
            check_overwrite(fname);
            fd_na = fopen(fname, "wb");
            if(!fd_na) std_err();
        }

        printf("- set input folder: %s\n", fdir);
        if(chdir(fdir) < 0) std_err();

        if(deflateInit2(&z, Z_BEST_COMPRESSION, Z_DEFLATED, 15, Z_BEST_COMPRESSION, Z_DEFAULT_STRATEGY) != Z_OK) {
            printf("\nError: zlib initialization error\n");
            exit(1);
        }

        strcpy(filedir, ".");
        recursive_dir(filedir);
        tot = add_files(NULL, 0, NULL, NULL, NULL); // create the NI file

        fclose(fd_ni);
        fclose(fd_na);
        goto quit;
    }

    printf("- EXTRACT mode\n");
    if(inflateInit2(&z, 15) != Z_OK) {
        printf("\nError: zlib initialization error\n");
        exit(1);
    }

    if(!stricmp(ext, ".z")) {
        printf("- open compressed file %s\n", fname);
        fd = fopen(fname, "rb");
        if(!fd) std_err();
        fstat(fileno(fd), &xstat);
        name = fname;

        *ext = 0;   // remove .Z

        myfr(fd, &crc,  sizeof(crc));
        myfr(fd, &size, sizeof(size));
        zsize = xstat.st_size - 8;

        myalloc(&in,  zsize, &insz);
        myalloc(&out, size,  &outsz);

        myfr(fd, in, zsize);
        unzip(in, zsize, out, size);
        if(crc != crc32(0, out, size)) printf("  bad CRC!!!\n");
        dumpa(name, out, size);
        tot++;

        goto quit;
    }

    if(numfile) {
        printf("- create configuration file %s\n", numfile);
        check_overwrite(numfile);
        fd_num = fopen(numfile, "wb");
        if(!fd_num) std_err();
    }

        /* NI */

    if(operation == NIENCRYPT) {
        read_ni(fname, ext, &nni, &nni_info, &nni_names, NIENCRYPT);

        strcpy(ext, ".ni_encrypted");
        printf("- create decrypted index file %s\n", fname);
        check_overwrite(fname);
        fd = fopen(fname, "wb");
        if(!fd) std_err();
        myfw(fd, nni, sizeof(nni_t));
        myfw(fd, nni_info, sizeof(nni_info_t) * nni->tot);
        myfw(fd, nni_names, nni->namesz);
        tot++;
        goto quit;
    }

    read_ni(fname, ext, &nni, &nni_info, &nni_names, NIDECRYPT);

    if(operation == NIDECRYPT) {
        strcpy(ext, ".ni_decrypted");
        printf("- create decrypted index file %s\n", fname);
        check_overwrite(fname);
        fd = fopen(fname, "wb");
        if(!fd) std_err();
        myfw(fd, nni, sizeof(nni_t));
        myfw(fd, nni_info, sizeof(nni_info_t) * nni->tot);
        myfw(fd, nni_names, nni->namesz);
        tot++;
        goto quit;
    }

        /* NA */

    if(operation == EXTRACT) {
        strcpy(ext, ".na");
        printf("- open data file %s\n", fname);
        fd = fopen(fname, "rb");
        if(!fd) std_err();

        printf("- set output folder: %s\n", fdir);
        if(chdir(fdir) < 0) std_err();
    }
    if(operation == LISTONLY) {
        printf("\n"
            "  number   data_size  offset   nameoff  filename\n"
            "------------------------------------------------\n");
    }

    for(i = 0; i < nni->tot; i++) {
        if(nni_info[i].nameoff > nni->namesz) continue;  // useless
        name = nni_names + nni_info[i].nameoff;

        if(fd_num) fprintf(fd_num, "%08x %s\r\n", nni_info[i].num, name);
        if(operation == LISTONLY) {
            printf("  %08x %-10u %08x %08x %s\n",
                nni_info[i].num, nni_info[i].size, nni_info[i].offset, nni_info[i].nameoff, name);
            continue;
        }
        if(fseek(fd, nni_info[i].offset, SEEK_SET)) std_err();

        ext = strrchr(name, '.');
        if(ext && !stricmp(ext, ".z")) {
            if(nni_info[i].size < 8) continue;
            *ext = 0;   // remove .Z
            if(force_wildcard && (check_wildcard(name, force_wildcard) < 0)) continue;

            myfr(fd, &crc,  sizeof(crc));
            myfr(fd, &size, sizeof(size));
            zsize = nni_info[i].size - 8;

            myalloc(&in,  zsize, &insz);
            myalloc(&out, size,  &outsz);

            myfr(fd, in, zsize);
            unzip(in, zsize, out, size);
            if(crc != crc32(0, out, size)) printf("  bad CRC!!!\n");
        } else {
            if(force_wildcard && (check_wildcard(name, force_wildcard) < 0)) continue;

            size = nni_info[i].size;
            myalloc(&out, size,  &outsz);
            myfr(fd, out, size);
        }
        dumpa(name, out, size);
        tot++;
    }

quit:
    if(fd_num) fclose(fd_num);
    if(fd) fclose(fd);
    inflateEnd(&z);
    printf("\n- %d files extracted/archived\n", tot);
    return(0);
}



int check_wildcard(u8 *fname, u8 *wildcard) {
    u8      *f,
            *w,
            *a;

    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



u32 get_num_list(u8 *fname) {
    int     i;

    if(!num_list) return(-1);

    for(i = 0; num_list[i].fname; i++) {
        if(!stricmp(fname, num_list[i].fname)) return(num_list[i].num);
    }
    return(-1);
}



void build_num_list(u8 *numfile) {
    FILE    *fd;
    u32     num;
    int     i;
    u8      buff[PATHMAX + 16],
            *fname,
            *p;

    num_list = NULL;
    if(!numfile) {
        printf("\n"
            "Error: you must specify the -n FD option for rebuilding the archive correctly\n"
            "       this option must be used when extracting or listing the origina NA\n"
            "       archive for getting the configurations of the file and then when you\n"
            "       want to rebuild it (the file must be the same used in the extraction)\n");
        exit(1);
    }

    printf("- open number file %s\n", numfile);
    fd = fopen(numfile, "rb");
    if(!fd) std_err();

    for(i = 0; fgets(buff, sizeof(buff), fd); i++);
    i++;  // the last must be NULLed!
    num_list = malloc(sizeof(num_list_t) * i);
    if(!num_list) std_err();

    rewind(fd);
    fname = buff + 8 + 1;

    for(i = 0; fgets(buff, sizeof(buff), fd);) {
        for(p = buff; *p && (*p != '\n') && (*p != '\r'); p++);
        *p = 0;

        if(sscanf(buff, "%08x", &num) != 1) continue;
        num_list[i].num   = num;
        num_list[i].fname = strdup(fname);
        i++;
    }
    num_list[i].num   = 0;
    num_list[i].fname = NULL;

    fclose(fd);
}



void read_ni(u8 *fname, u8 *ext, nni_t **nni_ret, nni_info_t **nni_info_ret, u8 **nni_names_ret, int crypt) {
    nni_info_t  *nni_info;
    nni_t   *nni;
    FILE    *fd;
    u8      *nni_names;

    strcpy(ext, ".ni");
    printf("- open index file %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();

    nni = malloc(sizeof(nni_t));
    if(!nni) std_err();

    myfr(fd, nni, sizeof(nni_t));
    if(nni->sign != 0x00494e4e) {
        printf("\nError: wrong signature (0x%08x)\n", nni->sign);
        exit(1);
    }

    nni_info = malloc(sizeof(nni_info_t) * nni->tot);
    if(!nni_info) std_err();
    myfr(fd, nni_info, sizeof(nni_info_t) * nni->tot);
    if(crypt == NIDECRYPT) {
        ys_ni_decrypt((void *)nni_info, sizeof(nni_info_t) * nni->tot);
    } else {
        ys_ni_encrypt((void *)nni_info, sizeof(nni_info_t) * nni->tot);
    }

    nni_names = malloc(nni->namesz);
    if(!nni_names) std_err();
    myfr(fd, nni_names, nni->namesz);
    if(crypt == NIDECRYPT) {
        ys_ni_decrypt(nni_names, nni->namesz);
    } else {
        ys_ni_encrypt(nni_names, nni->namesz);
    }

    fclose(fd);
    *nni_ret        = nni;
    *nni_info_ret   = nni_info;
    *nni_names_ret  = nni_names;
}



void check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return;
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') exit(1);
}



int add_files(u8 *fname, u32 fsize, nni_t *nni_in, nni_info_t *nni_info_in, u8 *nni_names_in) {
    static nni_info_t   *nni_info   = NULL;
    static int  nni_info_size       = 0,
                names_size          = 0,
                tot                 = 0,
                insz                = 0,
                outsz               = 0;
    static u32  offset              = 0,
                nameoff             = 0;
    static u8   *names              = NULL,
                *in                 = NULL,
                *out                = NULL;
    nni_t   nni;
    FILE    *fd;
    u32     num,
            crc,
            infosz;
    int     i,
            dozip,
            nlen,
            zlen,
            exists                  = 0,
            tot_bck                 = 0,
            nameoff_bck             = 0;

    if(!fname) {
        if(nni_in && nni_info_in && nni_names_in) {
            nni_info        = nni_info_in;
            nni_info_size   = nni_in->tot;
            names_size      = nni_in->namesz;
            tot             = nni_in->tot;
            offset          = ftell(fd_na);
            nameoff         = nni_in->namesz;
            names           = nni_names_in;
            return(0);
        }

        printf("- build NI index\n");

        infosz = sizeof(nni_info_t) * tot;
        ys_ni_encrypt((void *)nni_info, infosz);
        ys_ni_encrypt(names, nameoff);

        nni.sign   = 0x00494e4e;
        nni.tot    = tot;
        nni.namesz = nameoff;
        nni.zero   = 0;

        myfw(fd_ni, &nni, sizeof(nni_t));
        myfw(fd_ni, (void *)nni_info, infosz);
        myfw(fd_ni, names, nameoff);

        free(nni_info);
        free(names);
        free(in);
        free(out);
        return(tot);
    }

    if(force_wildcard && (check_wildcard(fname, force_wildcard) < 0)) return(tot);

    printf("  %-10u %s\n", fsize, fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();

    nlen = strlen(fname) + 3;
    if((nameoff + nlen) >= names_size) {
        names_size += nlen + 500000;
        names = realloc(names, names_size);
        if(!names) std_err();
    }
    dozip = 0;
    toupperz(fname, names + nameoff, dozip);
    num = get_num_list(names + nameoff);
    if(num == -1) {
        dozip = 1;
        toupperz(fname, names + nameoff, dozip);
        num = get_num_list(names + nameoff);
        if(num == -1) {
            printf("\n"
                "Error: this file has not been found in the configuration file you have\n"
                "       specified with the -n option, I need the exact configuration of this\n"
                "       file if you want to append or rebuild the archive\n");
            exit(1);
        }
    }

    if(dozip) {
        myalloc(&in, fsize, &insz);
        myfr(fd, in, fsize);
        myalloc(&out, ZLIBMAX(fsize), &outsz);
        zlen = zip(in, fsize, out, outsz);
    } else {
        myalloc(&out, fsize, &outsz);
        myfr(fd, out, fsize);
        zlen = 0;   // avoid the warning
    }
    fclose(fd);

    if(operation == APPEND) {
        for(i = 0; i < tot; i++) {
            if(!strcmp(names + nameoff, names + nni_info[i].nameoff)) break;
        }
        if(i < tot) {
            printf("  file %s found at position %d\n", names + nni_info[i].nameoff, i);
            exists = 1;
            tot_bck = tot;
            tot = i;
            nameoff_bck = nameoff;
            nameoff = nni_info[i].nameoff;
        }
    }

    if(tot >= nni_info_size) {
        nni_info_size += 16384;
        nni_info = realloc(nni_info, sizeof(nni_info_t) * nni_info_size);
        if(!nni_info) std_err();
    }
    nni_info[tot].num     = num;
    nni_info[tot].size    = dozip ? (8 + zlen) : fsize;
    nni_info[tot].offset  = offset;
    nni_info[tot].nameoff = nameoff;

    if(dozip) {
        crc = crc32(0, in, fsize);
        myfw(fd_na, &crc,   sizeof(crc));
        myfw(fd_na, &fsize, sizeof(fsize));
        myfw(fd_na, out, zlen);
    } else {
        myfw(fd_na, out, fsize);
    }

    offset  += nni_info[tot].size;
    if(exists) {
        nameoff = nameoff_bck;
        tot     = tot_bck;
        return(tot);
    }
    nameoff += nlen;
    tot++;
    return(tot);
}



void toupperz(u8 *in, u8 *out, int addz) {
    u8      *i,
            *o;

    for(i = in, o = out; *i; i++, o++) {
        if(*i >= 0x80) {
            *o++ = *i++;
            *o   = *i;
            continue;
        }
        if(*i == '/') {
            *o = '\\';
            continue;
        }
        *o = toupper(*i);
    }
    if(addz) {
        *o++ = '.';
        *o++ = 'Z';
    }
    *o = 0;
}



int recursive_dir(u8 *filedir) {
    int     plen,
            ret     = -1;

#ifdef WIN32
    static int      winnt = -1;
    OSVERSIONINFO   osver;
    WIN32_FIND_DATA wfd;
    HANDLE  hFind;

    if(winnt < 0) {
        osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        GetVersionEx(&osver);
        if(osver.dwPlatformId >= VER_PLATFORM_WIN32_NT) {
            winnt = 1;
        } else {
            winnt = 0;
        }
    }

    plen = strlen(filedir);
    strcpy(filedir + plen, "\\*.*");
    plen++;

    if(winnt) { // required to avoid problems with Vista and Windows7!
        hFind = FindFirstFileEx(filedir, FindExInfoStandard, &wfd, FindExSearchNameMatch, NULL, 0);
    } else {
        hFind = FindFirstFile(filedir, &wfd);
    }
    if(hFind == INVALID_HANDLE_VALUE) return(0);
    do {
        if(!strcmp(wfd.cFileName, ".") || !strcmp(wfd.cFileName, "..")) continue;

        strcpy(filedir + plen, wfd.cFileName);

        if(wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
            if(recursive_dir(filedir) < 0) goto quit;
        } else {
            add_files(filedir + 2, wfd.nFileSizeLow, NULL, NULL, NULL);
        }
    } while(FindNextFile(hFind, &wfd));
    ret = 0;

quit:
    FindClose(hFind);
#else
    struct  stat    xstat;
    struct  dirent  **namelist;
    int     n,
            i;

    n = scandir(filedir, &namelist, NULL, NULL);
    if(n < 0) {
        if(stat(filedir, &xstat) < 0) {
            printf("**** %s", filedir);
            std_err();
        }
        add_files(filedir + 2, xstat.st_size, NULL, NULL, NULL);
        return(0);
    }

    plen = strlen(filedir);
    strcpy(filedir + plen, "/");
    plen++;

    for(i = 0; i < n; i++) {
        if(!strcmp(namelist[i]->d_name, ".") || !strcmp(namelist[i]->d_name, "..")) continue;
        strcpy(filedir + plen, namelist[i]->d_name);

        if(stat(filedir, &xstat) < 0) {
            printf("**** %s", filedir);
            std_err();
        }
        if(S_ISDIR(xstat.st_mode)) {
            if(recursive_dir(filedir) < 0) goto quit;
        } else {
            add_files(filedir + 2, xstat.st_size, NULL, NULL, NULL);
        }
        free(namelist[i]);
    }
    ret = 0;

quit:
    for(; i < n; i++) free(namelist[i]);
    free(namelist);
#endif
    filedir[plen - 1] = 0;
    return(ret);
}



void myalloc(u8 **data, unsigned wantsize, unsigned *currsize) {
    if(wantsize <= *currsize) return;
    *data = realloc(*data, wantsize);
    if(!*data) std_err();
    *currsize = wantsize;
}



void dumpa(u8 *fname, u8 *data, int datasz) {
    FILE    *fd;

    create_dir(fname);
    printf("  %-10u %s\n", datasz, fname);
    if(!overwrite) check_overwrite(fname);
    fd = fopen(fname, "wb");
    if(!fd) std_err();
    myfw(fd, data, datasz);
    fclose(fd);
}



void create_dir(u8 *name) {
    u8      *p,
            *l;

        // removed any security check since the files are ever officials

    for(p = name;; p = l + 1) {
        l = strchr(p, '\\');
        if(!l) l = strchr(p, '/');
        if(!l) break;
        *l = 0;

#ifdef WIN32
        mkdir(name);
#else
        mkdir(name, 0755);
#endif
        *l = PATHSLASH;
    }
}



void myfr(FILE *fd, void *data, unsigned size) {
    if(fread(data, 1, size, fd) == size) return;
    printf("\nError: incomplete input file, can't read %u bytes\n", size);
    exit(1);
}



void myfw(FILE *fd, void *data, unsigned size) {
    if(fwrite(data, 1, size, fd) == size) return;
    printf("\nError: problems during the writing of the output file\n");
    exit(1);
}



int zip(u8 *in, u32 insz, u8 *out, u32 outsz) {
    if(!insz) return(0);

    deflateReset(&z);

    z.next_in   = in;
    z.avail_in  = insz;
    z.next_out  = out;
    z.avail_out = outsz;
    if(deflate(&z, Z_FINISH) != Z_STREAM_END) {
        printf("\nError: the compressed output is wrong or incomplete\n");
        exit(1);
    }
    return(z.total_out);
}



int unzip(u8 *in, u32 insz, u8 *out, u32 outsz) {
    if(!outsz) return(0);

    inflateReset(&z);

    z.next_in   = in;
    z.avail_in  = insz;
    z.next_out  = out;
    z.avail_out = outsz;
    if(inflate(&z, Z_SYNC_FLUSH) != Z_STREAM_END) {
        printf("\nError: the compressed input is wrong or incomplete\n");
        exit(1);
    }
    return(z.total_out);
}



void ys_ni_decrypt(u8 *data, int datasz) {
    int     i,
            num;

    num = 0x7c53f961;
    for(i = 0; i < datasz; i++) {
        num *= 0x3d09;
        data[i] -= (num >> 16);
    }
}



void ys_ni_encrypt(u8 *data, int datasz) {
    int     i,
            num;

    num = 0x7c53f961;
    for(i = 0; i < datasz; i++) {
        num *= 0x3d09;
        data[i] += (num >> 16);
    }
}



void show_examples(void) {
    printf("\n"
        "Usage examples:\n"
        "  extract data.na:   ysext -n conf.txt c:\\ysf_win\\release\\data.na c:\\new_folder\n"
        "  extract the XSOs:  ysext -n conf.txt -f \"*.xso\" data.na new_folder\n"
        "  list data.na:      ysext -l c:\\ysf_win\\release\\data.na none\n"
        "  create conf.txt:   ysext -l -n conf.txt c:\\ysf_win\\release\\data.na none\n"
        "  unzip human.dds.z: ysext human.dds.z none\n"
        "  rebuild data.na:   ysext -r -n conf.txt c:\\data.na input_folder\n"
        "  add files:         ysext -a -n conf.txt c:\\data_1101.na input_folder\n"
        "  zip human.dds:     ysext -z human.dds none\n"
        "  decrypt data.ni:   ysext -x c:\\ysf_win\\release\\data.ni none\n"
        "\n"
        "the folder named \"none\" means that any folder you specify with that specific\n"
        "command is ignored because not needed, so you can specify any folder you want\n"
        "\n");
    exit(1);
}



void show_info(void) {
    printf("\n"
        "NA: files with this extension are archives which contain the files of the\n"
        "    game, almost all of the files archived in the NA packages are compressed\n"
        "    using the format described below for the Z files\n"
        "\n"
        "NI: index files which contains informations about the files archived in NA\n"
        "    like filenames, size, offset in the archive and their number.\n"
        "    the files (or some of them) indexed here can't be all compressed or\n"
        "    uncompressed and have also a number referred to their original position,\n"
        "    that's why the -n option is NEEDED for rebuilding the archives correctly\n"
        "\n"
        "Z:  the format for the zipped files which contains the CRC of the uncompressed\n"
        "    file followed by its uncompressed size and the zipped data\n"
        "\n");
    exit(1);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


